-------------------------------------------------------------------------------
-- Particle Physics Detector Electronics Support 
-- University of Wisconsin
-- Lawrence Berkeley National Laboratory (c) 1999
-- ReadOutDriver Electronics
--
-------------------------------------------------------------------------------
-- Filename: top.vhd alias "PixelFormatterFPGA"
-- Description:
--
-------------------------------------------------------------------------------
-- Structure: 
-------------------------------------------------------------------------------
-- Timing:
--    *The Engine FPGA runs at 40MHz => clk40
--    *The timing structure of all engine code will be:
--      1.  Perform all logic operations on positive edge of clk40
--      2.  Perform all memory operations on positive edge of clk40
-------------------------------------------------------------------------------
-- Author: John Joseph
-- Board Engineer: 
-- History: 2003 JMJ First version
--
-- Changes:
--
-- Notes:
--    1) ALL signals are positive logic!
--      ... that is, the vhdl code and test benches do not conform to 
--          to the top entity signal names.
-------------------------------------------------------------------------------

-------------------------------------------------------------------------------
-- LIBRARY INCLUDES
-------------------------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;  -- needed for logic operations
use IEEE.std_logic_arith.all; -- needed for +/- operations
use IEEE.std_logic_unsigned.all;

-- pragma translate_off
library UNISIM;
use unisim.all;
-- pragma translate_on

-------------------------------------------------------------------------------
-- PORT DECLARATION
-------------------------------------------------------------------------------
entity top_fpix is               -- FormatterFPGA
  port(
    clk_in                        : in    std_logic; -- clk40 input
    rst_n_in                      : in    std_logic; -- asynchronous global reset
    link_data_in                  : in    std_logic_vector(0 to 11); -- serial data from detector modules
    master_not_slave_in           : in    std_logic; -- ='1' trigger counter enabled, ='0' pass through
    l1_trigger_pulse_in           : in    std_logic; -- 1 clk40 pulse to trigger_counter, how many times to pass the token
    token_in                      : in    std_logic; --
    mode_bits_in                  : in    std_logic_vector(11 downto 0); -- 12 wire time multiplexed mode bits from rrif
    modebits_fifo_wen_n_in        : in    std_logic; -- modebit fifos writein strobe: strobe 0 - zero bits, strobe 1 - one bits
    modebits_fifo_rst_n_in        : in    std_logic; -- modebit fifos reset strobe
    hold_output_in                : in    std_logic; -- halt readout
    show_trailer_flags_in         : in    std_logic; -- input from rodcontroller to display the trapped trailer flags
    formatter_writes_trailer_out  : out   std_logic_vector(11 downto 0); -- this chip's trapped trailer flags
    formatter_bus_strb_n_in       : in    std_logic; -- read_enable write_strobe for internal register access
    formatter_bus_rnw_in          : in    std_logic; -- ='1' read out of registers, ='0' write into registers
    formatter_bus_hwsb_in         : in    std_logic; -- Half Word Select Bit
    formatter_bus_ds_in           : in    std_logic; -- Data Strobe for half word wr
    formatter_bus_adr_in          : in    std_logic_vector( 7 downto 0); -- [7..4] which channel, [3..0] which register
    formatter_bus_data_inout      : inout std_logic_vector(15 downto 0); -- inout data tofrom registers
    formatter_bus_ack_n_out       : out   std_logic; -- register/data transfer acknowledge signal
    header_trailer_limit_t_out    : out   std_logic; -- at least one of the channels has reached this configureable limit
    rodbusy_limit_t_out           : out   std_logic; -- at least one of the channels has reached this configureable limit
    time_out_error_out            : out   std_logic; -- configureable time out has occurred on the active channel
    data_overflow_error_out       : out   std_logic; -- configureable overflow has occurred on the active channel
    condensed_mode_out            : out   std_logic; -- 
    modebits_fifo_ef_n_out        : out   std_logic; -- at least one of the readout mode fifos is empty
    modebits_fifo_ff_n_out        : out   std_logic; -- at least one of the readout mode fifos is full 
    data_valid_out                : out   std_logic; -- ='1' fifo_data_out should be read as valid data
    fifo_data_out                 : out   std_logic_vector(31 downto 0); -- 
    token_out                     : out   std_logic; -- 
    link_number_has_token_out     : out   std_logic_vector( 3 downto 0); -- 
    this_chip_has_token_out       : out   std_logic; -- active low
    spare_dout                    : out   std_logic_vector( 1 downto 0); -- 
    link_data_mux_out             : out   std_logic;
    fmt_type_in                   : in    std_logic_vector( 1 downto 0);
    rod_type_out                  : out   std_logic;
    test_out                      : out   std_logic_vector( 7 downto 0);  -- 
    spare_pin                     : in    std_logic_vector( 7 downto 2);  -- 
    spare_pout                    : out   std_logic_vector( 1 downto 0)   -- 
    );
end top_fpix; 

architecture rtl of top_fpix is

--------------------------------------------------------------------------
-- SIGNAL DECLARATION
--------------------------------------------------------------------------
-- pad connection signals

 signal rst_n_p : std_logic;
 signal rst_i   : std_logic;

 signal clk40_in_p         : std_logic;
 signal dll_clk40_out      : std_logic;
 signal dll_clk40_locked_i : std_logic;
 signal clk40_g            : std_logic;

 signal link_data_in_p                  : std_logic_vector(0 to 11);
 signal link_data_i                     : std_logic_vector(3 downto 0);
 signal link_data_i_simu                : std_logic_vector(3 downto 0);
 signal master_not_slave_in_p           : std_logic;
 signal l1_trigger_pulse_in_p           : std_logic;
 signal token_in_p                      : std_logic;
 signal mode_bits_in_p                  : std_logic_vector(11 downto 0);
 signal modebits_fifo_wen_n_in_p        : std_logic;
 signal modebits_fifo_rst_n_in_p        : std_logic;
 signal hold_output_in_i                : std_logic;
 signal hold_output_in_t                : std_logic;
 signal hold_output_in_p                : std_logic;
 signal show_trailer_flags_in_p         : std_logic;
 signal show_trailer_flags_t            : std_logic;
 signal formatter_writes_trailer_out_p  : std_logic_vector(11 downto 0);
 signal formatter_bus_strb_i            : std_logic;
 signal formatter_bus_strb_n_in_p       : std_logic;
 signal formatter_bus_hwsb_in_p         : std_logic;
 signal formatter_bus_hwsb_i            : std_logic;
 signal formatter_bus_ds_in_p           : std_logic;
 signal formatter_bus_ds_i              : std_logic;
 signal formatter_bus_rnw_in_p          : std_logic;
 signal formatter_bus_rnw_i             : std_logic;
 signal formatter_bus_adr_in_p          : std_logic_vector( 7 downto 0);
 signal formatter_bus_data_in_p         : std_logic_vector(15 downto 0);
 signal formatter_bus_data_out_p        : std_logic_vector(15 downto 0);
 signal formatter_bus_ack_n_out_p       : std_logic;
 signal formatter_bus_tbuf_drv_i        : std_logic;
 signal ht_limit_t                      : std_logic;
 signal rb_limit_t                      : std_logic;
 signal time_out_error_out_p            : std_logic;
 signal data_overflow_error_out_p       : std_logic;
 signal condensed_mode_out_p            : std_logic;
 signal mb_fifo_ef_i                    : std_logic;
 signal mb_fifo_ff_i                    : std_logic;
 signal modebits_fifo_ef_n_out_p        : std_logic;
 signal modebits_fifo_ff_n_out_p        : std_logic;
 signal data_valid_out_p                : std_logic;
 signal data_valid_out_i                : std_logic;
 signal fifo_data_out_p                 : std_logic_vector(31 downto 0);
 signal token_out_p                     : std_logic;
 signal link_number_has_token_out_p     : std_logic_vector( 3 downto 0);
 signal chip_has_token_n_p              : std_logic;
 signal chip_has_token_i                : std_logic;
 signal test_out_p                      : std_logic_vector( 7 downto 0);
 signal spare_pin_p                     : std_logic_vector( 7 downto 0);
 signal spare_dout_p                    : std_logic_vector( 1 downto 0);
 signal gnd_i                           : std_logic;

type   fifo_data_array_type is array (0 to 3) of std_logic_vector(31 downto 0); --
signal link_fifo_data_i  : fifo_data_array_type; --
signal debug_fifo_data_i : fifo_data_array_type; --
signal data_valid_i      : std_logic_vector( 3 downto 0);
signal occ_count_i       : std_logic_vector(51 downto 0); --

signal data_link_output_sel_i : std_logic_vector(3 downto 0);
signal mux_link_data_i : std_logic;

signal trigger_count_i  : std_logic_vector(9 downto 0);

signal normal_not_raw_i       : std_logic_vector(11 downto 0);
signal header_trailer_error_i : std_logic_vector(11 downto 0);
signal rod_busy_error_i       : std_logic_vector(11 downto 0);
signal time_out_error_i       : std_logic_vector(11 downto 0);
signal data_overflow_error_i  : std_logic_vector(11 downto 0);

signal timeout_limit_i        : unsigned(31 downto 0);
signal overflow_limit_i       : unsigned(15 downto 0);
signal header_trailer_limit_i : unsigned(15 downto 0);
signal rod_busy_limit_i       : unsigned(15 downto 0);
signal link_enabled_i         : std_logic_vector(11 downto 0);
signal ecr_link_flush_i       : std_logic_vector(11 downto 0);
signal config_mode_i          : std_logic_vector(11 downto 0);
signal edge_mode_i            : std_logic_vector(11 downto 0);
signal fifo_ren_i             : std_logic_vector(11 downto 0);
signal link_fifo_ef_i         : std_logic_vector(11 downto 0);
signal link_fifo_ff_i         : std_logic_vector(11 downto 0);

signal roc_mb_ren_i           : std_logic;
signal rb_mb_ren_i            : std_logic;
signal mb_fifo_ren_i          : std_logic;
signal mb_ready_i             : std_logic;
signal link_mb_i              : std_logic_vector(23 downto 0);

signal decoder_wr_trailer_i   : std_logic_vector(11 downto 0);
signal readout_state_i        : std_logic_vector( 3 downto 0);

signal token_out_i            : std_logic;
signal num_accepts_i          : std_logic_vector(15 downto 0);
signal roc_ready_i            : std_logic;
signal link_fifo_select_i     : std_logic_vector( 3 downto 0);
signal rb_link_fifo_data_i    : std_logic_vector(31 downto 0);
signal rb_link_fifo_data_o    : std_logic_vector(31 downto 0);
signal link_fifo_rd_i         : std_logic;
signal link_fifo_wr_i         : std_logic;
signal fmt_type_i             : std_logic_vector( 1 downto 0);
signal fmt_type_p             : std_logic_vector( 1 downto 0);
signal fmt_type_n_p           : std_logic_vector( 1 downto 0);
signal rod_type_p             : std_logic;
signal fmt_number_i           : std_logic_vector( 1 downto 0);
signal na_count_i             : std_logic_vector(15 downto 0);
signal eoe_word_i             : std_logic_vector( 3 downto 0);
signal fifo_pause_i           : std_logic_vector( 3 downto 0);

signal link0_map_i : std_logic_vector( 3 downto 0);
signal link1_map_i : std_logic_vector( 3 downto 0);
signal link2_map_i : std_logic_vector( 3 downto 0);
signal link3_map_i : std_logic_vector( 3 downto 0);

signal simu_config_1 : std_logic_vector( 31 DOWNTO 0); -- configuration word for the simulator
signal simu_config_2 : std_logic_vector( 31 DOWNTO 0); -- configuration word for the simulator

signal l1a_i : std_logic;
signal bcr_i : std_logic;
signal ecr_i : std_logic;

signal tim_l1id_i       : std_logic_vector( 3 downto 0);
signal tim_last_l1id_i  : std_logic_vector(23 downto 0);

signal dbg_count_ctrl_i     : std_logic_vector( 1 downto 0);
signal mod0_resync_count_i  : std_logic_vector(31 DOWNTO 0);
signal mod0_hitword_count_i : std_logic_vector(31 DOWNTO 0);
signal mod0_event_count_i   : std_logic_vector(31 DOWNTO 0);
signal mod0_error_count_i   : std_logic_vector(31 DOWNTO 0);
signal mod0_dproc_count_i   : std_logic_vector(31 DOWNTO 0);
signal mod0_clk_count_i     : std_logic_vector(31 DOWNTO 0);
signal mod1_resync_count_i  : std_logic_vector(31 DOWNTO 0);
signal mod1_hitword_count_i : std_logic_vector(31 DOWNTO 0);
signal mod1_event_count_i   : std_logic_vector(31 DOWNTO 0);
signal mod1_error_count_i   : std_logic_vector(31 DOWNTO 0);
signal mod1_dproc_count_i   : std_logic_vector(31 DOWNTO 0);
signal mod1_clk_count_i     : std_logic_vector(31 DOWNTO 0);
signal mod2_resync_count_i  : std_logic_vector(31 DOWNTO 0);
signal mod2_hitword_count_i : std_logic_vector(31 DOWNTO 0);
signal mod2_event_count_i   : std_logic_vector(31 DOWNTO 0);
signal mod2_error_count_i   : std_logic_vector(31 DOWNTO 0);
signal mod2_dproc_count_i   : std_logic_vector(31 DOWNTO 0);
signal mod2_clk_count_i     : std_logic_vector(31 DOWNTO 0);
signal mod3_resync_count_i  : std_logic_vector(31 DOWNTO 0);
signal mod3_hitword_count_i : std_logic_vector(31 DOWNTO 0);
signal mod3_event_count_i   : std_logic_vector(31 DOWNTO 0);
signal mod3_error_count_i   : std_logic_vector(31 DOWNTO 0);
signal mod3_dproc_count_i   : std_logic_vector(31 DOWNTO 0);
signal mod3_clk_count_i     : std_logic_vector(31 DOWNTO 0);


-------------------------------------------------------------------------------
-- COMPONENT DECLARATION
-------------------------------------------------------------------------------

component register_block
  port (
    clk40_in               : in  std_logic; -- clk40 input
    rst_n_in               : in  std_logic; -- async global reset
    frm_bus_strb_in        : in  std_logic; --
    frm_bus_rnw_in         : in  std_logic; --
    frm_bus_hwsb_in        : in  std_logic; --
    frm_bus_ds_in          : in  std_logic; --
    frm_bus_addr_in        : in  std_logic_vector( 7 downto 0); --
    frm_bus_data_in        : in  std_logic_vector(15 downto 0); --
    frm_bus_data_out       : out std_logic_vector(15 downto 0); --
    frm_bus_ack_out        : out std_logic; --
    normal_not_raw         : in  std_logic_vector(11 downto 0); --
    header_trailer_error   : in  std_logic_vector(11 downto 0); --
    rod_busy_error         : in  std_logic_vector(11 downto 0); --
    time_out_error         : in  std_logic_vector(11 downto 0); --
    data_overflow_error    : in  std_logic_vector(11 downto 0); --
    occupancy_count_link0  : in  std_logic_vector(12 downto 0); --
    occupancy_count_link1  : in  std_logic_vector(12 downto 0); --
    occupancy_count_link2  : in  std_logic_vector(12 downto 0); --
    occupancy_count_link3  : in  std_logic_vector(12 downto 0); --
    mode_bits              : in  std_logic_vector(23 downto 0); --
    trigger_count          : in  std_logic_vector( 9 downto 0); --
    hold_output            : in  std_logic; --
    frm_dll_locked         : in  std_logic; --
    master_not_slave       : in  std_logic; --
    timeout_limit          : out unsigned(31 downto 0); --
    overflow_limit         : out unsigned(15 downto 0); --
    header_trailer_limit   : out unsigned(15 downto 0); --
    rod_busy_limit         : out unsigned(15 downto 0); --
    link_enabled           : out std_logic_vector(11 downto 0); --
    ecr_link_flush         : out std_logic_vector(11 downto 0); --
    config_mode            : out std_logic_vector(11 downto 0); --
    edge_mode              : out std_logic_vector(11 downto 0); --
    mb_fifo_rd             : out std_logic;
    mb_fifo_ef             : in  std_logic; --
    mb_fifo_ff             : in  std_logic; --
    mb_rst_n_in            : in  std_logic;
    active_link_status     : in  std_logic_vector( 3 downto 0); --
    data_link_out_mux_sel  : out std_logic_vector( 3 downto 0); --
    chip_has_token_in      : in  std_logic;
    readout_status_in      : in  std_logic_vector( 3 downto 0);
    ht_limit_en_n_out      : out std_logic;
    rb_limit_en_n_out      : out std_logic;
    do_error_out           : out std_logic;
    num_accepts_out        : out std_logic_vector(15 downto 0);
    fmt_type_in            : in  std_logic_vector( 1 downto 0);
    fmt_type_out           : out std_logic_vector( 1 downto 0);
    rod_type_out           : out std_logic;
    spare_pin_reg_in       : in  std_logic_vector( 7 downto 0);
    link_fifo_select       : out std_logic_vector( 3 downto 0);
    link_fifo_data_in      : in  std_logic_vector(31 downto 0);
    link_fifo_data_out     : out std_logic_vector(31 downto 0); 
    link_fifo_rd           : out std_logic;
    link_fifo_wr           : out std_logic;
    num_accepts_in         : in  std_logic_vector(15 downto 0);
    fmt_number_in          : in  std_logic_vector( 1 downto 0);
    link0_map_out          : out std_logic_vector( 3 downto 0);
    link1_map_out          : out std_logic_vector( 3 downto 0);
    link2_map_out          : out std_logic_vector( 3 downto 0);
    link3_map_out          : out std_logic_vector( 3 downto 0);
    boc_scan_data_in       : in  std_logic_vector( 3 downto 0);
    simu_config_1          : out std_logic_vector(31 downto 0);
    simu_config_2          : out std_logic_vector(31 downto 0);
    pres_l1id_out          : out std_logic_vector( 3 downto 0);
    last_l1id_out          : out std_logic_vector(23 downto 0);
    l1a_in                 : in  std_logic;
    bcr_in                 : in  std_logic;
    ecr_in                 : in  std_logic;
    dbg_count_ctrl_out     : out std_logic_vector( 1 downto 0);
    mod0_resync_count_in   : in  std_logic_vector(31 DOWNTO 0);
    mod0_hitword_count_in  : in  std_logic_vector(31 DOWNTO 0);
    mod0_event_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod0_error_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod0_dproc_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod0_clk_count_in      : in  std_logic_vector(31 DOWNTO 0);
    mod1_resync_count_in   : in  std_logic_vector(31 DOWNTO 0);
    mod1_hitword_count_in  : in  std_logic_vector(31 DOWNTO 0);
    mod1_event_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod1_error_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod1_dproc_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod1_clk_count_in      : in  std_logic_vector(31 DOWNTO 0);
    mod2_resync_count_in   : in  std_logic_vector(31 DOWNTO 0);
    mod2_hitword_count_in  : in  std_logic_vector(31 DOWNTO 0);
    mod2_event_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod2_error_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod2_dproc_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod2_clk_count_in      : in  std_logic_vector(31 DOWNTO 0);
    mod3_resync_count_in   : in  std_logic_vector(31 DOWNTO 0);
    mod3_hitword_count_in  : in  std_logic_vector(31 DOWNTO 0);
    mod3_event_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod3_error_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod3_dproc_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod3_clk_count_in      : in  std_logic_vector(31 DOWNTO 0)
    );   
end component;

component mb_data_decode
  port(
    clk_in           : in  std_logic; -- clk40 input
    rst_n_in         : in  std_logic; -- asynchronous global reset
    mode_bits_in     : in  std_logic_vector(11 downto 0);
    mb_fifo_wen_n_in : in  std_logic; -- mb fifos wr strobe: active low
    mb_fifo_ren_in   : in  std_logic; -- mb fifos rd strobe: active high
    mb_fifo_rst_n_in : in  std_logic; -- modebit fifos reset strobe
    mode_bits_out    : out std_logic_vector(23 downto 0);
    mb_ready_out     : out std_logic; 
    mb_fifo_ef_n_out : out std_logic; -- readout mode fifo is empty
    mb_fifo_ff_n_out : out std_logic  -- readout mode fifo is full 
    );
end component; 

component trailer_to_fifo_detect
  port(
    clk_in              : in  std_logic; -- 
    rst_n_in            : in  std_logic; --
    show_trailer_flags  : in  std_logic; --
    trailer_to_fifo_in  : in  std_logic_vector(11 downto 0);
    trailer_to_fifo_out : out std_logic_vector(11 downto 0)
    );
end component;

component tctg  -- Trigger Counter/Token Generator
  port(
    clk_in              : in  std_logic;
    rst_n_in            : in  std_logic;
    master_not_slave_in : in  std_logic;
    roc_ready_in        : in  std_logic;
    l1_trigger_pulse_in : in  std_logic;
    token_in            : in  std_logic;
    token_out           : out std_logic;
    trigger_count_out   : out std_logic_vector(9 downto 0)
    );
end component;

component quad_link_formatter
  port(
    clk_in                 : in  std_logic; -- 40 MHz clock
    rst_n_in               : in  std_logic; -- Powerup global reset
    link_enable_in         : in  std_logic_vector( 3 DOWNTO 0);
    ecr_link_flush         : in  std_logic_vector( 3 DOWNTO 0);
    decode_type_in         : in  std_logic_vector( 1 DOWNTO 0);
    config_mode_in         : in  std_logic_vector( 3 DOWNTO 0); -- Not Used
    half_clk_err_mask_in   : in  std_logic_vector( 3 DOWNTO 0);                                                                
    link_data_in           : in  std_logic_vector( 3 DOWNTO 0); -- front end serial data
    ht_limit_in            : in  unsigned(15 DOWNTO 0);         -- Header/Trailer Limit value
    rod_busy_limit_in      : in  unsigned(15 DOWNTO 0);         -- ROD Busy Limit value
    num_accepts_in         : in  std_logic_vector(15 DOWNTO 0); -- Number of accepts in Event Data per link
    link_to_error_in       : in  std_logic_vector( 3 DOWNTO 0);  -- Link Time out
    fifo_ren_in            : in  std_logic_vector( 3 DOWNTO 0); -- fifo REN
    fifo_pause_in          : in  std_logic_vector( 3 DOWNTO 0);  -- fifo pause
    normal_not_raw_out     : out std_logic_vector( 3 DOWNTO 0); -- 1/0 = normal/raw data mode
    decoder_wr_trailer_out : out std_logic_vector( 3 DOWNTO 0); -- trailer being written to FIFO
    ht_limit_out           : out std_logic_vector( 3 DOWNTO 0);
    rod_busy_limit_out     : out std_logic_vector( 3 DOWNTO 0);
    fifo_ef_out            : out std_logic_vector( 3 DOWNTO 0);
    fifo_ff_out            : out std_logic_vector( 3 DOWNTO 0);
    occ_count_out          : out std_logic_vector(51 DOWNTO 0);
    fifo0_data_out         : out std_logic_vector(31 DOWNTO 0); -- Output data word
    fifo1_data_out         : out std_logic_vector(31 DOWNTO 0); -- Output data word
    fifo2_data_out         : out std_logic_vector(31 DOWNTO 0); -- Output data word
    fifo3_data_out         : out std_logic_vector(31 DOWNTO 0); -- Output data word
    data_valid_out         : out std_logic_vector( 3 DOWNTO 0); -- Last word of Event
    eoe_word_out           : out std_logic_vector( 3 DOWNTO 0);
    num_accepts_out        : out std_logic_vector(15 downto 0);
    ecr_in                 : in  std_logic;
    tim_l1id_in            : in  std_logic_vector( 3 DOWNTO 0);
    last_l1id_in           : in  std_logic_vector(23 DOWNTO 0);
    dbg_count_ctrl_in      : in  std_logic_vector( 1 downto 0);
    mod0_resync_count_out  : out std_logic_vector(31 DOWNTO 0);
    mod0_hitword_count_out : out std_logic_vector(31 DOWNTO 0);
    mod0_event_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod0_error_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod0_dproc_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod0_clk_count_out     : out std_logic_vector(31 DOWNTO 0);
    mod1_resync_count_out  : out std_logic_vector(31 DOWNTO 0);
    mod1_hitword_count_out : out std_logic_vector(31 DOWNTO 0);
    mod1_event_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod1_error_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod1_dproc_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod1_clk_count_out     : out std_logic_vector(31 DOWNTO 0);
    mod2_resync_count_out  : out std_logic_vector(31 DOWNTO 0);
    mod2_hitword_count_out : out std_logic_vector(31 DOWNTO 0);
    mod2_event_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod2_error_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod2_dproc_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod2_clk_count_out     : out std_logic_vector(31 DOWNTO 0);
    mod3_resync_count_out  : out std_logic_vector(31 DOWNTO 0);
    mod3_hitword_count_out : out std_logic_vector(31 DOWNTO 0);
    mod3_event_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod3_error_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod3_dproc_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod3_clk_count_out     : out std_logic_vector(31 DOWNTO 0)
    );
end component ; 

component fifo_readout_controller
  port(
    clk_in                  : in  std_logic;
    rst_n_in                : in  std_logic;
    enable_link_in          : in  std_logic_vector( 3 downto 0);
    config_mode_in          : in  std_logic_vector( 3 downto 0);
    token_in                : in  std_logic;
    hold_output_in          : in  std_logic;
    fifo_timeout_limit_in   : in  unsigned(31 downto 0); -- 
    data_overflow_limit_in  : in  unsigned(15 downto 0); -- 
    link0_fifo_data_in      : in  std_logic_vector(31 downto 0);
    link1_fifo_data_in      : in  std_logic_vector(31 downto 0);
    link2_fifo_data_in      : in  std_logic_vector(31 downto 0);
    link3_fifo_data_in      : in  std_logic_vector(31 downto 0);
    data_valid_in           : in  std_logic_vector( 3 downto 0);
    link_fifo_empty_in      : in  std_logic_vector( 3 downto 0);
    link_fifo_full_in       : in  std_logic_vector( 3 downto 0);
    mb_ready_in             : in  std_logic;
    mb_fifo_empty_in        : in  std_logic;
    mode_bits_in            : in  std_logic_vector(23 downto 0); 
    formatter_fifo_data_out : out std_logic_vector(31 downto 0);
    link_num_has_token_out  : out std_logic_vector( 3 downto 0);
    timeout_error_out       : out std_logic;
    condensed_mode_out      : out std_logic;
    data_valid_out          : out std_logic;
    chip_has_token_out      : out std_logic;
    timeout_error_status    : out std_logic_vector( 3 downto 0);
    data_overflow_status    : out std_logic_vector( 3 downto 0);
    mb_ren_out              : out std_logic;
    fifo_ren_out            : out std_logic_vector( 3 downto 0);
    fifo_pause_out          : out std_logic_vector( 3 downto 0);
    token_out               : out std_logic;
    readout_state_out       : out std_logic_vector( 3 downto 0);
    roc_ready_out           : out std_logic;
    decode_type_in          : in  std_logic_vector( 1 downto 0);
    na_count_in             : in  std_logic_vector(15 DOWNTO 0); -- Number of accepts in Event Data per link
    eoe_word_in             : in  std_logic_vector( 3 downto 0);
    ecr_in                  : in  std_logic
    );
end component;

component fe_cmd_decoder
  port (
    clk_in      : in  std_logic; -- clk40 input
    rst_n_in    : in  std_logic; -- asynchronous global reset
    enable_in   : in  std_logic;
    ser_data_in : in  std_logic; 
    l1a_out     : out std_logic;
    bcr_out     : out std_logic;
    ecr_out     : out std_logic
    );
end component;

component pixel_simulator
  port(
    clk_in         : in  std_logic;  -- 40 MHz clock
    rst_n_in       : in  std_logic;  -- Powerup global reset
    config_in_1    : in  std_logic_vector( 31 DOWNTO 0);  -- configuration word for the simulator
    config_in_2    : in  std_logic_vector( 31 DOWNTO 0);  -- configuration word for the simulator
    level1_trigger : in  std_logic;  -- Level 1 Trigger Input
    BCR            : in  std_logic;  -- Bunch Counter Reset
    ECR            : in  std_logic;  -- Event Counter Reset	 
    inlinks        : in  std_logic_vector(3 DOWNTO 0);  -- Real Input links
    outlinks       : out std_logic_vector(3 DOWNTO 0)   -- output for real or encoded data
    );
end component; 

--------------------------------------------------------------------------
-- XILINX PARTS SPECIFIC COMPONENTS/SIGNALS
--------------------------------------------------------------------------

component STARTUP_VIRTEX 
  port(
    GSR : in std_logic;
    GTS : in std_logic;
    CLK : in std_logic
    );
end component;


--------------------------------------------------------------------------
-- PAD DECLARATIONS
--------------------------------------------------------------------------
component IBUFG
  port (
    I : in  STD_LOGIC ;
    O : out STD_LOGIC
    );
end component;

component BUFG
  port (
    I : in  STD_LOGIC ;
    O : out STD_LOGIC
    );
end component;

component IBUF
  port (
    I : in  STD_LOGIC ;
    O : out STD_LOGIC
    );
end component;

component OBUF
  port (
    I : in  STD_LOGIC ;
    O : out STD_LOGIC
    );
end component;

component OBUFT
  port (
    I : in  STD_LOGIC ;
    O : out STD_LOGIC ;
    T : in  STD_LOGIC
    );
end component;

component IOBUF
  port (
    I  : in    STD_LOGIC ;
    O  : out   STD_LOGIC ;
    IO : inout STD_LOGIC ;
    T  : in    STD_LOGIC
    );
end component;

component CLKDLL
  port (
    CLK0   : out STD_LOGIC ;
    CLK90  : out STD_LOGIC ;
    CLK180 : out STD_LOGIC ;
    CLK270 : out STD_LOGIC ;
    CLK2X  : out STD_LOGIC ;
    CLKDV  : out STD_LOGIC ;
    LOCKED : out STD_LOGIC ;
    CLKIN  : in  STD_LOGIC ;
    CLKFB  : in  STD_LOGIC ;
    RST    : in  STD_LOGIC
    );
end component;

begin
-------------------------------------------------------------------------------
-- PAD INSTANTIATION
-------------------------------------------------------------------------------

clk_ibufg : IBUFG port map(
  I => clk_in, 
  O => clk40_in_p
  );

main_clk_dll: CLKDLL 
  port map(
    CLK0   => dll_clk40_out,
    CLK90  => open,
    CLK180 => open,
    CLK270 => open,
    CLK2X  => open,
    CLKDV  => open,
    LOCKED => dll_clk40_locked_i,
    CLKIN  => clk40_in_p,
    CLKFB  => clk40_g,
    RST    => gnd_i
    );

clk_bufg : BUFG port map(
  I => dll_clk40_out, 
  O => clk40_g
  );

rst_buf : IBUF  port map(
  I => rst_n_in, 
  O => rst_n_p
  );
rst_i <= NOT rst_n_p;

link_data_inputs : for i in 0 to 11 generate
  link_data_ibufs : IBUF
    port map(
      I => link_data_in(i), 
      O => link_data_in_p(i)
      );
end generate;

mns_ibuf : IBUF port map(
  I => master_not_slave_in, 
  O => master_not_slave_in_p
  );

l1_trig_ibuf : IBUF port map(
  I => l1_trigger_pulse_in, 
  O => l1_trigger_pulse_in_p
  );

token_ibuf : IBUF port map(
  I => token_in, 
  O => token_in_p
  );
  
token_obuf : OBUF port map(
  I => token_out_p, 
  O => token_out
  );

mb_data_inputs :  for i in 0 to 11 generate
  mb_data_ibufs : IBUF
    port map(
      I => mode_bits_in(i), 
      O => mode_bits_in_p(i)
      );
end generate;

mb_wen_n_ibuf : IBUF port map(
  I => modebits_fifo_wen_n_in, 
  O => modebits_fifo_wen_n_in_p
  );

mb_rst_n_ibuf : IBUF port map(
  I => modebits_fifo_rst_n_in, 
  O => modebits_fifo_rst_n_in_p
  );

mb_ef_n_obuf : OBUFT
  port map(
    I => gnd_i, 
    O => modebits_fifo_ef_n_out,
    T => hold_output_in_t        -- indicates to RCF that EFB has issued a pause
    );                           -- tristate == '1', drive to 0 == '0'

mb_fifo_ef_i <= NOT modebits_fifo_ef_n_out_p;

mb_ff_n_obuf : OBUFT
  port map(
    I => gnd_i, 
    O => modebits_fifo_ff_n_out,
    T => modebits_fifo_ff_n_out_p
    );
mb_fifo_ff_i <= NOT modebits_fifo_ff_n_out_p;

hold_ouput_ibuf : IBUF port map(
  I => hold_output_in, 
  O => hold_output_in_p
  );

stflags_ibuf : IBUF port map(
  I => show_trailer_flags_in, 
  O => show_trailer_flags_in_p
  );

stflags_outputs :  for i in 0 to 11 generate
  stflags_obufts : OBUFT
    port map(
      I => formatter_writes_trailer_out_p(i),
      O => formatter_writes_trailer_out(i),
      T => show_trailer_flags_t
      );
end generate;

fb_strbn_ibuf : IBUF  port map(
  I => formatter_bus_strb_n_in,
  O => formatter_bus_strb_n_in_p
  );

fb_rnw_ibuf : IBUF port map(
  I => formatter_bus_rnw_in, 
  O => formatter_bus_rnw_in_p
  );

fb_hwsb_ibuf : IBUF port map(
  I => formatter_bus_hwsb_in, 
  O => formatter_bus_hwsb_in_p
  );

fb_ds_ibuf : IBUF port map(
  I => formatter_bus_ds_in, 
  O => formatter_bus_ds_in_p
  );

fb_addr_inputs : for i in 0 to  7 generate
  fb_addr_ibufs : IBUF
    port map(
      I => formatter_bus_adr_in(i), 
      O => formatter_bus_adr_in_p(i)
      );
end generate;

fb_data_io : for i in 0 to 15 generate
  fb_data_iobufs : IOBUF
    port map(
      I  => formatter_bus_data_out_p(i),
      O  => formatter_bus_data_in_p(i),
      IO => formatter_bus_data_inout(i), 
      T  => formatter_bus_tbuf_drv_i
      );
end generate;

fb_ackn_obuft : OBUFT
  port map(
    I => gnd_i,
    O => formatter_bus_ack_n_out,
    T => formatter_bus_ack_n_out_p
    );

chip_has_token_obuf : OBUF port map(
  I => chip_has_token_n_p, 
  O => this_chip_has_token_out
  );

data_overflow_obuft : OBUFT
  port map(
    I => data_overflow_error_out_p,
    O => data_overflow_error_out,
    T => chip_has_token_n_p
    );

fifo_data_outputs : for i in 0 to 31 generate
  fifo_data_obufts : OBUFT
    port map(
      I => fifo_data_out_p(i), 
      O => fifo_data_out(i), 
      T => chip_has_token_n_p
      );
end generate;

link_has_token_outputs : for i in 0 to 3 generate
  link_has_token_obufts : OBUFT
    port map(
      I => link_number_has_token_out_p(i), 
      O => link_number_has_token_out(i),
      T => chip_has_token_n_p
      );
end generate;

timeout_error_obuft : OBUFT
  port map(
    I => time_out_error_out_p,
    O => time_out_error_out,
    T => chip_has_token_n_p
    );
                              
condensed_mode_out_obuft : OBUFT
  port map(
    I => condensed_mode_out_p,
    O => condensed_mode_out, 
    T => chip_has_token_n_p
    );

data_valid_obuft : OBUFT
  port map(
    I => data_valid_out_p, 
    O => data_valid_out,
    T => chip_has_token_n_p
    );
data_valid_out_p <= NOT data_valid_out_i;

htlimit_obuft : OBUFT
  port map(
    I => gnd_i, 
    O => header_trailer_limit_t_out,
    T => ht_limit_t
    );

rblimit_obuft : OBUFT
  port map(
    I => gnd_i,
    O => rodbusy_limit_t_out,
    T => rb_limit_t
    );

test_outputs :  for i in 0 to 7 generate
  test_obufs : OBUF
    port map(
      I => test_out_p(i), 
      O => test_out(i)
      );
end generate;

spare_outputs :  for i in 0 to 1 generate
  spare_dout_obufts : OBUFT
    port map(
      I => spare_dout_p(i), 
      O => spare_dout(i),
      T => chip_has_token_n_p
      );
end generate;

spare_pin_block : for i in 2 to 7 generate 
  spare_pin_ibuf : IBUF
    port map (
      I => spare_pin(i),
      O => spare_pin_p(i)
      );
end generate;

spare_pout_block : for i in 0 to 1 generate 
  spare_pin_obuf : OBUF
    port map (
      I => spare_pin_p(i),
      O => spare_pout(i)
      );
end generate;

data_link_mux_obuf : OBUF 
  port map (
    I => mux_link_data_i,
    O => link_data_mux_out
    );

rod_type_obuft : OBUFT 
  port map (
    I => gnd_i,
    O => rod_type_out,
    T => rod_type_p
    );

fmt_type_block : for i in 0 to 1 generate 
  fmt_type_ibuf : IBUF
    port map (
      I => fmt_type_in(i),
      O => fmt_type_n_p(i)
      );
end generate;


--------------------------------------------------------------------------
-- XILINX PARTS SPECIFIC COMPONENTS/SIGNALS
--------------------------------------------------------------------------

U1 : STARTUP_VIRTEX 
  port map (
    GSR => rst_i,
    GTS => rst_i,
    CLK => clk40_g
    );

--------------------------------------------------------------------------
-- COMPONENT INSTANTIATION
--------------------------------------------------------------------------
U2 : register_block
  port map (
    clk40_in               => clk40_g,
    rst_n_in               => rst_n_p,
    frm_bus_strb_in        => formatter_bus_strb_i,
    frm_bus_rnw_in         => formatter_bus_rnw_i,
    frm_bus_hwsb_in        => formatter_bus_hwsb_i,
    frm_bus_ds_in          => formatter_bus_ds_i,
    frm_bus_addr_in        => formatter_bus_adr_in_p,
    frm_bus_data_in        => formatter_bus_data_in_p,
    frm_bus_data_out       => formatter_bus_data_out_p,
    frm_bus_ack_out        => formatter_bus_ack_n_out_p,
    normal_not_raw         => normal_not_raw_i,
    header_trailer_error   => header_trailer_error_i,
    rod_busy_error         => rod_busy_error_i,
    time_out_error         => time_out_error_i,
    data_overflow_error    => data_overflow_error_i,
    occupancy_count_link0  => occ_count_i(12 downto  0),
    occupancy_count_link1  => occ_count_i(25 downto 13),
    occupancy_count_link2  => occ_count_i(38 downto 26),
    occupancy_count_link3  => occ_count_i(51 downto 39),
    mode_bits              => link_mb_i,
    trigger_count          => trigger_count_i,
    hold_output            => hold_output_in_i,
    frm_dll_locked         => dll_clk40_locked_i,
    master_not_slave       => master_not_slave_in_p,
    timeout_limit          => timeout_limit_i,
    overflow_limit         => overflow_limit_i,
    header_trailer_limit   => header_trailer_limit_i,
    rod_busy_limit         => rod_busy_limit_i,
    link_enabled           => link_enabled_i,
    ecr_link_flush         => ecr_link_flush_i,
    config_mode            => config_mode_i,
    edge_mode              => edge_mode_i,
    mb_fifo_rd             => rb_mb_ren_i,
    mb_fifo_ef             => mb_fifo_ef_i,
    mb_fifo_ff             => mb_fifo_ff_i,
    mb_rst_n_in            => modebits_fifo_rst_n_in_p,
    active_link_status     => link_number_has_token_out_p,
    data_link_out_mux_sel  => data_link_output_sel_i,
    chip_has_token_in      => chip_has_token_i,
    readout_status_in      => readout_state_i,
    ht_limit_en_n_out      => ht_limit_t,
    rb_limit_en_n_out      => rb_limit_t,
    do_error_out           => data_overflow_error_out_p, 
    num_accepts_out        => num_accepts_i,
    fmt_type_in            => fmt_type_p,
    fmt_type_out           => fmt_type_i,
    rod_type_out           => rod_type_p,
    spare_pin_reg_in       => spare_pin_p,
    link_fifo_select       => link_fifo_select_i,
    link_fifo_data_in      => rb_link_fifo_data_i,
    link_fifo_data_out     => rb_link_fifo_data_o,
    link_fifo_rd           => link_fifo_rd_i,
    link_fifo_wr           => link_fifo_wr_i,
    num_accepts_in         => na_count_i,
    fmt_number_in          => fmt_number_i,
    link0_map_out          => link0_map_i,
    link1_map_out          => link1_map_i,
    link2_map_out          => link2_map_i,
    link3_map_out          => link3_map_i,
    boc_scan_data_in       => link_data_i(3 downto 0),
    simu_config_1          => simu_config_1,
    simu_config_2          => simu_config_2,
    pres_l1id_out          => tim_l1id_i,
    last_l1id_out          => tim_last_l1id_i,
    l1a_in                 => l1a_i,
    bcr_in                 => bcr_i,
    ecr_in                 => ecr_i,
    dbg_count_ctrl_out     => dbg_count_ctrl_i,
    mod0_resync_count_in   => mod0_resync_count_i,
    mod0_hitword_count_in  => mod0_hitword_count_i,
    mod0_event_count_in    => mod0_event_count_i,
    mod0_error_count_in    => mod0_error_count_i,
    mod0_dproc_count_in    => mod0_dproc_count_i,
    mod0_clk_count_in      => mod0_clk_count_i,
    mod1_resync_count_in   => mod1_resync_count_i,
    mod1_hitword_count_in  => mod1_hitword_count_i,
    mod1_event_count_in    => mod1_event_count_i,
    mod1_error_count_in    => mod1_error_count_i,
    mod1_dproc_count_in    => mod1_dproc_count_i,
    mod1_clk_count_in      => mod1_clk_count_i,
    mod2_resync_count_in   => mod2_resync_count_i,
    mod2_hitword_count_in  => mod2_hitword_count_i,
    mod2_event_count_in    => mod2_event_count_i,
    mod2_error_count_in    => mod2_error_count_i,
    mod2_dproc_count_in    => mod2_dproc_count_i,
    mod2_clk_count_in      => mod2_clk_count_i,
    mod3_resync_count_in   => mod3_resync_count_i,
    mod3_hitword_count_in  => mod3_hitword_count_i,
    mod3_event_count_in    => mod3_event_count_i,
    mod3_error_count_in    => mod3_error_count_i,
    mod3_dproc_count_in    => mod3_dproc_count_i,
    mod3_clk_count_in      => mod3_clk_count_i
    );   

U3 : mb_data_decode
  port map (
    clk_in                 => clk40_g,
    rst_n_in               => rst_n_p,
    mode_bits_in           => mode_bits_in_p,
    mb_fifo_wen_n_in       => modebits_fifo_wen_n_in_p,
    mb_fifo_ren_in         => mb_fifo_ren_i,
    mb_fifo_rst_n_in       => rst_n_p,  --modebits_fifo_rst_n_in_p,
    mode_bits_out          => link_mb_i,
    mb_ready_out           => mb_ready_i,
    mb_fifo_ef_n_out       => modebits_fifo_ef_n_out_p,
    mb_fifo_ff_n_out       => modebits_fifo_ff_n_out_p
    );

U4 : trailer_to_fifo_detect
  port map (
    clk_in                 => clk40_g,
    rst_n_in               => rst_n_p,
    show_trailer_flags     => show_trailer_flags_in_p,
    trailer_to_fifo_in     => decoder_wr_trailer_i,
    trailer_to_fifo_out    => formatter_writes_trailer_out_p
    );

U5 : tctg
  port map (
    clk_in                 => clk40_g,
    rst_n_in               => rst_n_p,
    master_not_slave_in    => master_not_slave_in_p,
    roc_ready_in           => roc_ready_i,
    l1_trigger_pulse_in    => l1_trigger_pulse_in_p,
    token_in               => token_in_p,
    token_out              => token_out_i,
    trigger_count_out      => trigger_count_i
    );

U6 : quad_link_formatter
    port map (
      clk_in                 => clk40_g,
      rst_n_in               => rst_n_p,
      link_enable_in         => link_enabled_i(3 downto 0),
      ecr_link_flush         => ecr_link_flush_i(3 downto 0),
      decode_type_in         => fmt_type_i,
      config_mode_in         => config_mode_i(3 downto 0),
      half_clk_err_mask_in   => edge_mode_i(3 downto 0),
--*******************************************************************
-- Comment out to compile with the FE simulatorsimulator
--      link_data_in           => link_data_i(3 downto 0),
-- Uncomment to compile with the FE simulator
      link_data_in           => link_data_i_simu(3 downto 0),
--*******************************************************************
      ht_limit_in            => header_trailer_limit_i,
      rod_busy_limit_in      => rod_busy_limit_i,
      link_to_error_in       => time_out_error_i(3 downto 0),
      fifo_ren_in            => fifo_ren_i(3 downto 0),
      fifo_pause_in          => fifo_pause_i,
      num_accepts_in         => num_accepts_i,
      normal_not_raw_out     => normal_not_raw_i(3 downto 0),
      decoder_wr_trailer_out => decoder_wr_trailer_i(3 downto 0),
      ht_limit_out           => header_trailer_error_i(3 downto 0),
      rod_busy_limit_out     => rod_busy_error_i(3 downto 0),
      fifo_ef_out            => link_fifo_ef_i(3 downto 0),
      fifo_ff_out            => link_fifo_ff_i(3 downto 0),
      occ_count_out          => occ_count_i,
      fifo0_data_out         => link_fifo_data_i(0),
      fifo1_data_out         => link_fifo_data_i(1),
      fifo2_data_out         => link_fifo_data_i(2),
      fifo3_data_out         => link_fifo_data_i(3),
      data_valid_out         => data_valid_i,
      eoe_word_out           => eoe_word_i,
      num_accepts_out        => na_count_i,
      ecr_in                 => ecr_i,
      tim_l1id_in            => tim_l1id_i,
      last_l1id_in           => tim_last_l1id_i,
      dbg_count_ctrl_in      => dbg_count_ctrl_i,
      mod0_resync_count_out  => mod0_resync_count_i,
      mod0_hitword_count_out => mod0_hitword_count_i,
      mod0_event_count_out   => mod0_event_count_i,
      mod0_error_count_out   => mod0_error_count_i,
      mod0_dproc_count_out   => mod0_dproc_count_i,
      mod0_clk_count_out     => mod0_clk_count_i,
      mod1_resync_count_out  => mod1_resync_count_i,
      mod1_hitword_count_out => mod1_hitword_count_i,
      mod1_event_count_out   => mod1_event_count_i,
      mod1_error_count_out   => mod1_error_count_i,
      mod1_dproc_count_out   => mod1_dproc_count_i,
      mod1_clk_count_out     => mod1_clk_count_i,
      mod2_resync_count_out  => mod2_resync_count_i,
      mod2_hitword_count_out => mod2_hitword_count_i,
      mod2_event_count_out   => mod2_event_count_i,
      mod2_error_count_out   => mod2_error_count_i,
      mod2_dproc_count_out   => mod2_dproc_count_i,
      mod2_clk_count_out     => mod2_clk_count_i,
      mod3_resync_count_out  => mod3_resync_count_i,
      mod3_hitword_count_out => mod3_hitword_count_i,
      mod3_event_count_out   => mod3_event_count_i,
      mod3_error_count_out   => mod3_error_count_i,
      mod3_dproc_count_out   => mod3_dproc_count_i,
      mod3_clk_count_out     => mod3_clk_count_i
      );

U7 : fifo_readout_controller
  port map(
    clk_in                  => clk40_g,
    rst_n_in                => rst_n_p,
    enable_link_in          => link_enabled_i(3 downto 0),
    config_mode_in          => config_mode_i(3 downto 0),
    token_in                => token_out_i,
    hold_output_in          => hold_output_in_i,
    fifo_timeout_limit_in   => timeout_limit_i,
    data_overflow_limit_in  => overflow_limit_i,
    link0_fifo_data_in      => link_fifo_data_i(0),
    link1_fifo_data_in      => link_fifo_data_i(1),
    link2_fifo_data_in      => link_fifo_data_i(2),
    link3_fifo_data_in      => link_fifo_data_i(3),
    data_valid_in           => data_valid_i,
    link_fifo_empty_in      => link_fifo_ef_i(3 downto 0),
    link_fifo_full_in       => link_fifo_ff_i(3 downto 0),
    mb_ready_in             => mb_ready_i,
    mb_fifo_empty_in        => mb_fifo_ef_i,
    mode_bits_in            => link_mb_i,
    formatter_fifo_data_out => fifo_data_out_p,
    link_num_has_token_out  => link_number_has_token_out_p,
    timeout_error_out       => time_out_error_out_p,
    condensed_mode_out      => condensed_mode_out_p,
    data_valid_out          => data_valid_out_i,
    chip_has_token_out      => chip_has_token_i,
    timeout_error_status    => time_out_error_i(3 downto 0),
    data_overflow_status    => data_overflow_error_i(3 downto 0),
    mb_ren_out              => roc_mb_ren_i,
    fifo_ren_out            => fifo_ren_i(3 downto 0),
    fifo_pause_out          => fifo_pause_i,
    token_out               => token_out_p,
    readout_state_out       => readout_state_i,
    roc_ready_out           => roc_ready_i,
    decode_type_in          => fmt_type_i,
    na_count_in             => na_count_i,
    eoe_word_in             => eoe_word_i,
    ecr_in                  => ecr_i
    );                          

-- FE Command Decoder for RCF FE CMD signals
U8 : fe_cmd_decoder
  port map(
    clk_in      => clk40_g,
    rst_n_in    => rst_n_p,
    enable_in   => rst_n_p,  --simu_config_1(0),
    ser_data_in => modebits_fifo_rst_n_in_p,  -- This signal is now a copy of the FE CMD signal
    l1a_out     => l1a_i,
    bcr_out     => bcr_i,
    ecr_out     => ecr_i
    );

-- Link Input Simulator
U9 : pixel_simulator
  port map(
    clk_in         => clk40_g,
    rst_n_in       => rst_n_p,
    config_in_1    => simu_config_1,
    config_in_2    => simu_config_2,
    level1_trigger => l1a_i,
    BCR            => bcr_i,
    ECR            => ecr_i,
    inlinks        => link_data_i,
    outlinks       => link_data_i_simu
    );

-------------------------------------------------------------------------------
-- SIGNAL DEFINITIONS
-------------------------------------------------------------------------------
gnd_i <= '0';

mb_fifo_ren_i      <= roc_mb_ren_i OR rb_mb_ren_i;
chip_has_token_n_p <= NOT chip_has_token_i;

-------------------------------------------------------------------------------
-- PROCESS DECLARATION
-------------------------------------------------------------------------------

fmt_number_select : process (
  clk40_g,
  rst_n_p
  )
begin
  if (rst_n_p = '0') then
    fmt_number_i <= (others => '0');
  elsif (clk40_g'event AND clk40_g = '1') then
    if (master_not_slave_in_p = '1') then
      fmt_number_i <= (others => '0');
      spare_pin_p(1 downto 0) <= "01";
    else
      fmt_number_i <= spare_pin_p(3 downto 2);
      case spare_pin_p(3 downto 2) is
        when "01"   => spare_pin_p(1 downto 0) <= "10";
        when "10"   => spare_pin_p(1 downto 0) <= "11";
        when others => spare_pin_p(1 downto 0) <= "00";
      end case;
    end if;
  end if;
end process fmt_number_select;

input_link_reg : process (
  clk40_g,
  rst_n_p
  )
begin
  if (rst_n_p = '0') then
    link_data_i <= (others => '0');
  elsif (clk40_g'event AND clk40_g = '1') then
    case link0_map_i is
      when "0000" => link_data_i(0) <= link_data_in_p( 0);
      when "0001" => link_data_i(0) <= link_data_in_p( 1);
      when "0010" => link_data_i(0) <= link_data_in_p( 2);
      when "0011" => link_data_i(0) <= link_data_in_p( 3);
      when "0100" => link_data_i(0) <= link_data_in_p( 4);
      when "0101" => link_data_i(0) <= link_data_in_p( 5);
      when "0110" => link_data_i(0) <= link_data_in_p( 6);
      when "0111" => link_data_i(0) <= link_data_in_p( 7);
      when "1000" => link_data_i(0) <= link_data_in_p( 8);
      when "1001" => link_data_i(0) <= link_data_in_p( 9);
      when "1010" => link_data_i(0) <= link_data_in_p(10);
      when "1011" => link_data_i(0) <= link_data_in_p(11);
      when others => link_data_i(0) <= '0';
    end case;

    case link1_map_i is
      when "0000" => link_data_i(1) <= link_data_in_p( 0);
      when "0001" => link_data_i(1) <= link_data_in_p( 1);
      when "0010" => link_data_i(1) <= link_data_in_p( 2);
      when "0011" => link_data_i(1) <= link_data_in_p( 3);
      when "0100" => link_data_i(1) <= link_data_in_p( 4);
      when "0101" => link_data_i(1) <= link_data_in_p( 5);
      when "0110" => link_data_i(1) <= link_data_in_p( 6);
      when "0111" => link_data_i(1) <= link_data_in_p( 7);
      when "1000" => link_data_i(1) <= link_data_in_p( 8);
      when "1001" => link_data_i(1) <= link_data_in_p( 9);
      when "1010" => link_data_i(1) <= link_data_in_p(10);
      when "1011" => link_data_i(1) <= link_data_in_p(11);
      when others => link_data_i(1) <= '0';
    end case;

    case link2_map_i is
      when "0000" => link_data_i(2) <= link_data_in_p( 0);
      when "0001" => link_data_i(2) <= link_data_in_p( 1);
      when "0010" => link_data_i(2) <= link_data_in_p( 2);
      when "0011" => link_data_i(2) <= link_data_in_p( 3);
      when "0100" => link_data_i(2) <= link_data_in_p( 4);
      when "0101" => link_data_i(2) <= link_data_in_p( 5);
      when "0110" => link_data_i(2) <= link_data_in_p( 6);
      when "0111" => link_data_i(2) <= link_data_in_p( 7);
      when "1000" => link_data_i(2) <= link_data_in_p( 8);
      when "1001" => link_data_i(2) <= link_data_in_p( 9);
      when "1010" => link_data_i(2) <= link_data_in_p(10);
      when "1011" => link_data_i(2) <= link_data_in_p(11);
      when others => link_data_i(2) <= '0';
    end case;

    case link3_map_i is
      when "0000" => link_data_i(3) <= link_data_in_p( 0);
      when "0001" => link_data_i(3) <= link_data_in_p( 1);
      when "0010" => link_data_i(3) <= link_data_in_p( 2);
      when "0011" => link_data_i(3) <= link_data_in_p( 3);
      when "0100" => link_data_i(3) <= link_data_in_p( 4);
      when "0101" => link_data_i(3) <= link_data_in_p( 5);
      when "0110" => link_data_i(3) <= link_data_in_p( 6);
      when "0111" => link_data_i(3) <= link_data_in_p( 7);
      when "1000" => link_data_i(3) <= link_data_in_p( 8);
      when "1001" => link_data_i(3) <= link_data_in_p( 9);
      when "1010" => link_data_i(3) <= link_data_in_p(10);
      when "1011" => link_data_i(3) <= link_data_in_p(11);
      when others => link_data_i(3) <= '0';
    end case;
  end if;
end process input_link_reg;

data_link_mux : process (
  data_link_output_sel_i,
  link_data_in_p,
  rb_limit_t
  )
begin
  case data_link_output_sel_i is
    when "0000" => mux_link_data_i <= link_data_in_p(0);    -- to spare_pin(7)
    when "0001" => mux_link_data_i <= link_data_in_p(1);
    when "0010" => mux_link_data_i <= link_data_in_p(2);
    when "0011" => mux_link_data_i <= link_data_in_p(3);
    when "0100" => mux_link_data_i <= link_data_in_p(4);
    when "0101" => mux_link_data_i <= link_data_in_p(5);
    when "0110" => mux_link_data_i <= link_data_in_p(6);
    when "0111" => mux_link_data_i <= link_data_in_p(7);
    when "1000" => mux_link_data_i <= link_data_in_p(8);
    when "1001" => mux_link_data_i <= link_data_in_p(9);
    when "1010" => mux_link_data_i <= link_data_in_p(10);
    when "1011" => mux_link_data_i <= link_data_in_p(11);
    when "1111" => mux_link_data_i <= NOT rb_limit_t;
    when others => mux_link_data_i <= '0';
  end case;
end process data_link_mux;

show_trailer_flags_control : process (
  clk40_g,
  rst_n_p
  )
begin
  if (rst_n_p = '0') then
    show_trailer_flags_t <= '1';
  elsif (clk40_g'event AND clk40_g = '1') then
    show_trailer_flags_t <= NOT show_trailer_flags_in_p; -- since HIGH means high_impedance and 
  end if;                                                --  show_trailer_flags_in_p is a positive logic signal.
end process show_trailer_flags_control;

frm_bus_output_enable : process (
  clk40_g,
  rst_n_p
  )
begin
  if (rst_n_p = '0') then
    formatter_bus_tbuf_drv_i <= '1';
  elsif (clk40_g'event AND clk40_g = '1') then
    if (formatter_bus_strb_i = '1' AND formatter_bus_rnw_i = '1') then
      formatter_bus_tbuf_drv_i <= '0';
    else
      formatter_bus_tbuf_drv_i <= '1';
    end if;
  end if;
end process frm_bus_output_enable;

test_port : process (
  clk40_g,
  rst_n_p
  )
begin
  if (rst_n_p = '0') then
    test_out_p <= (others => '0');
  elsif (clk40_g'event AND clk40_g = '1') then
    test_out_p(0) <= l1_trigger_pulse_in_p;
    test_out_p(1) <= data_valid_out_p;
    test_out_p(2) <= chip_has_token_n_p;
    test_out_p(3) <= fifo_data_out_p(29);
    test_out_p(4) <= fifo_data_out_p(30);
    test_out_p(5) <= fifo_data_out_p(31);
    test_out_p(6) <= fifo_data_out_p(14);
    test_out_p(7) <= fifo_data_out_p(15);
  end if;
end process test_port;
  
signal_register : process (
  clk40_g,
  rst_n_p
  )
begin
  if (rst_n_p = '0') then
    hold_output_in_i     <= '0';
    spare_dout_p         <= (others => '0');
    fmt_type_p           <= (others => '0');
    formatter_bus_strb_i <= '0';
    formatter_bus_hwsb_i <= '0';
    formatter_bus_ds_i   <= '0';
    formatter_bus_rnw_i  <= '0';
  elsif (clk40_g'event AND clk40_g = '1') then
    hold_output_in_i     <= hold_output_in_p;
    hold_output_in_t     <= NOT hold_output_in_p;
    spare_dout_p         <= ht_limit_t & rb_limit_t;
    fmt_type_p           <= NOT fmt_type_n_p(1) & NOT fmt_type_n_p(0);
    formatter_bus_strb_i <= NOT formatter_bus_strb_n_in_p;
    formatter_bus_hwsb_i <= formatter_bus_hwsb_in_p;
    formatter_bus_ds_i   <= formatter_bus_ds_in_p;
    formatter_bus_rnw_i  <= formatter_bus_rnw_in_p;
  end if;
end process signal_register;

debug_fifo_mux : process (
  clk40_g,
  rst_n_p
  )
begin
  if (rst_n_p = '0') then
    debug_fifo_data_i(0) <= (others => '0');
    debug_fifo_data_i(1) <= (others => '0');
    debug_fifo_data_i(2) <= (others => '0');
    debug_fifo_data_i(3) <= (others => '0');
  elsif (clk40_g'event AND clk40_g = '1') then
    case link_fifo_select_i(3 downto 1) is
      when "000" => 
        if (link_fifo_wr_i = '1') then
          debug_fifo_data_i(0) <= rb_link_fifo_data_o;
--        debug_fifo_wr_i      <= link_fifo_select_i(0) & NOT link_fifo_select_i(0);
        elsif (link_fifo_rd_i = '1') then
          rb_link_fifo_data_i  <= debug_fifo_data_i(0);
--        debug_fifo_rd_i      <= link_fifo_select_i(0) & NOT link_fifo_select_i(0);
        end if;
      when "001" => 
        if (link_fifo_wr_i = '1') then
          debug_fifo_data_i(1) <= rb_link_fifo_data_o;
        elsif (link_fifo_rd_i = '1') then
          rb_link_fifo_data_i  <= debug_fifo_data_i(1);
        end if;
      when "010" => 
        if (link_fifo_wr_i = '1') then
          debug_fifo_data_i(2) <= rb_link_fifo_data_o;
        elsif (link_fifo_rd_i = '1') then
          rb_link_fifo_data_i  <= debug_fifo_data_i(2);
        end if;
      when "011" => 
        if (link_fifo_wr_i = '1') then
          debug_fifo_data_i(3) <= rb_link_fifo_data_o;
        elsif (link_fifo_rd_i = '1') then
          rb_link_fifo_data_i  <= debug_fifo_data_i(3);
        end if;
      when others => 
    end case;
  end if;
end process debug_fifo_mux;
  
end rtl;
