--------------------------------------------------------------------------
-- Particle Physics Detector Electronics Support 
-- University of Wisconsin
-- Lawrence Berkeley National Laboratory (c) 1999
-- SiROD ReadOutDriver Electronics
--------------------------------------------------------------------------
-- Filename: register_block.vhd
-- Description:
--   Register file for Formatter Register Bus access. Examine the comments
--	 specifically to discover the exact register definition.
--      ROD Bus Registers:
--        FMT0 == 0x00400000 to 0x004000FC
--        FMT1 == 0x00400400 to 0x004004FC
--        FMT2 == 0x00400800 to 0x004008FC
--        FMT3 == 0x00400C00 to 0x00400CFC
--        FMT4 == 0x00401000 to 0x004010FC
--        FMT5 == 0x00401400 to 0x004014FC
--        FMT6 == 0x00401800 to 0x004018FC
--        FMT7 == 0x00401C00 to 0x00401CFC
--------------------------------------------------------------------------
-- Structure:
--------------------------------------------------------------------------
-- Timing:
--    *The Engine FPGA runs at 40MHz => clk40
--    *The timing structure of all engine code will be:
--      1.  Perform all logic operations on positive edge of clk40
--      2.  Perform all memory operations on positive edge of clk40
--------------------------------------------------------------------------
-- Author: Johm M. Joseph
-- Board Engineer: John M. Joseph
-- History:
--------------------------------------------------------------------------
-- Notes:
--------------------------------------------------------------------------
--------------------------------------------------------------------------
-- LIBRARY INCLUDES
--------------------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;  -- needed for logic operations
use IEEE.std_logic_arith.all; -- needed for +/- operations
use IEEE.std_logic_unsigned.all; -- needed for +/- operations
--------------------------------------------------------------------------
-- PORT DECLARATION
--------------------------------------------------------------------------
entity register_block is
  port (
    clk40_in               : in  std_logic; -- clk40 input
    rst_n_in               : in  std_logic; -- async global reset
    frm_bus_strb_in        : in  std_logic; --
    frm_bus_rnw_in         : in  std_logic; --
    frm_bus_hwsb_in        : in  std_logic; --
    frm_bus_ds_in          : in  std_logic; --
    frm_bus_addr_in        : in  std_logic_vector( 7 downto 0); --
    frm_bus_data_in        : in  std_logic_vector(15 downto 0); --
    frm_bus_data_out       : out std_logic_vector(15 downto 0); --
    frm_bus_ack_out        : out std_logic; --
    normal_not_raw         : in  std_logic_vector(11 downto 0); --
    header_trailer_error   : in  std_logic_vector(11 downto 0); --
    rod_busy_error         : in  std_logic_vector(11 downto 0); --
    time_out_error         : in  std_logic_vector(11 downto 0); --
    data_overflow_error    : in  std_logic_vector(11 downto 0); --
    occupancy_count_link0  : in  std_logic_vector(12 downto 0); --
    occupancy_count_link1  : in  std_logic_vector(12 downto 0); --
    occupancy_count_link2  : in  std_logic_vector(12 downto 0); --
    occupancy_count_link3  : in  std_logic_vector(12 downto 0); --
    mode_bits              : in  std_logic_vector(23 downto 0); --
    trigger_count          : in  std_logic_vector( 9 downto 0); --
    hold_output            : in  std_logic; --
    frm_dll_locked         : in  std_logic; --
    master_not_slave       : in  std_logic; --
    timeout_limit          : out unsigned(31 downto 0); --
    overflow_limit         : out unsigned(15 downto 0); --
    header_trailer_limit   : out unsigned(15 downto 0); --
    rod_busy_limit         : out unsigned(15 downto 0); --
    link_enabled           : out std_logic_vector(11 downto 0); --
    ecr_link_flush         : out std_logic_vector(11 downto 0); --
    config_mode            : out std_logic_vector(11 downto 0); --
    edge_mode              : out std_logic_vector(11 downto 0); --
    mb_fifo_rd             : out std_logic;
    mb_fifo_ef             : in  std_logic; --
    mb_fifo_ff             : in  std_logic; --
    mb_rst_n_in            : in  std_logic;
    active_link_status     : in  std_logic_vector( 3 downto 0); --
    data_link_out_mux_sel  : out std_logic_vector( 3 downto 0); --
    chip_has_token_in      : in  std_logic;
    readout_status_in      : in  std_logic_vector( 3 downto 0);
    ht_limit_en_n_out      : out std_logic;
    rb_limit_en_n_out      : out std_logic;
    do_error_out           : out std_logic;
    num_accepts_out        : out std_logic_vector(15 downto 0);
    fmt_type_in            : in  std_logic_vector( 1 downto 0);
    fmt_type_out           : out std_logic_vector( 1 downto 0);
    rod_type_out           : out std_logic;
    spare_pin_reg_in       : in  std_logic_vector( 7 downto 0);
    link_fifo_select       : out std_logic_vector( 3 downto 0);
    link_fifo_data_in      : in  std_logic_vector(31 downto 0);
    link_fifo_data_out     : out std_logic_vector(31 downto 0); 
    link_fifo_rd           : out std_logic;
    link_fifo_wr           : out std_logic;
    num_accepts_in         : in  std_logic_vector(15 downto 0);
    fmt_number_in          : in  std_logic_vector( 1 downto 0);
    link0_map_out          : out std_logic_vector( 3 downto 0);
    link1_map_out          : out std_logic_vector( 3 downto 0);
    link2_map_out          : out std_logic_vector( 3 downto 0);
    link3_map_out          : out std_logic_vector( 3 downto 0);
    boc_scan_data_in       : in  std_logic_vector( 3 downto 0);
    simu_config_1          : out std_logic_vector(31 downto 0);
    simu_config_2          : out std_logic_vector(31 downto 0);
    pres_l1id_out          : out std_logic_vector( 3 downto 0);
    last_l1id_out          : out std_logic_vector(23 downto 0);
    l1a_in                 : in  std_logic;
    bcr_in                 : in  std_logic;
    ecr_in                 : in  std_logic;
    dbg_count_ctrl_out     : out std_logic_vector( 1 downto 0);
    mod0_resync_count_in   : in  std_logic_vector(31 DOWNTO 0);
    mod0_hitword_count_in  : in  std_logic_vector(31 DOWNTO 0);
    mod0_event_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod0_error_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod0_dproc_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod0_clk_count_in      : in  std_logic_vector(31 DOWNTO 0);
    mod1_resync_count_in   : in  std_logic_vector(31 DOWNTO 0);
    mod1_hitword_count_in  : in  std_logic_vector(31 DOWNTO 0);
    mod1_event_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod1_error_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod1_dproc_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod1_clk_count_in      : in  std_logic_vector(31 DOWNTO 0);
    mod2_resync_count_in   : in  std_logic_vector(31 DOWNTO 0);
    mod2_hitword_count_in  : in  std_logic_vector(31 DOWNTO 0);
    mod2_event_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod2_error_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod2_dproc_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod2_clk_count_in      : in  std_logic_vector(31 DOWNTO 0);
    mod3_resync_count_in   : in  std_logic_vector(31 DOWNTO 0);
    mod3_hitword_count_in  : in  std_logic_vector(31 DOWNTO 0);
    mod3_event_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod3_error_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod3_dproc_count_in    : in  std_logic_vector(31 DOWNTO 0);
    mod3_clk_count_in      : in  std_logic_vector(31 DOWNTO 0)
    );   
end register_block;

architecture rtl of register_block is
--------------------------------------------------------------------------
-- SIGNAL DECLARATION
--------------------------------------------------------------------------
signal edge_mode_i       : std_logic_vector(11 downto 0); --
signal config_mode_i     : std_logic_vector(11 downto 0); --
signal ecr_link_flush_i  : std_logic_vector(11 downto 0); --
signal link_enabled_i    : std_logic_vector(11 downto 0); --
signal timeout_limit_i   : std_logic_vector(31 downto 0); --
signal overflow_limit_i  : std_logic_vector(15 downto 0); --
signal ht_limit_i        : std_logic_vector(15 downto 0); --
signal rod_busy_limit_i  : std_logic_vector(15 downto 0); --
signal dlink_out_mux_i   : std_logic_vector( 3 downto 0); --
signal time_out_error_i  : std_logic_vector(11 downto 0); --
signal clr_to_status_i   : std_logic;
signal clr_to_reg_i      : std_logic;
signal dovr_error_i      : std_logic_vector(11 downto 0); --
signal clr_dovr_status_i : std_logic;
signal clr_dovr_reg_i    : std_logic;
signal normal_not_raw_i  : std_logic_vector(11 downto 0); --
signal clr_nnr_status_i  : std_logic;
signal clr_nnr_reg_i     : std_logic;
signal rbl_error_i       : std_logic_vector(11 downto 0); --
signal clr_rbl_status_i  : std_logic;
signal clr_rbl_reg_i     : std_logic;
signal htl_error_i       : std_logic_vector(11 downto 0); --
signal clr_htl_status_i  : std_logic;
signal clr_htl_reg_i     : std_logic;
signal num_accepts_i     : std_logic_vector(15 downto 0); --

type states is (
  idle, 
  acknowledge, 
  wait_cycle
  );
signal acknowledge_cycle_state : states;

signal en_link_fifo_rd_i  : std_logic;
signal en_link_fifo_wr_i  : std_logic;
signal link_fifo_select_i : std_logic_vector( 3 downto 0);
signal en_mb_fifo_rd_i    : std_logic;
signal mb_rst_n_i         : std_logic;
signal fmt_type_i         : std_logic_vector( 1 downto 0);
signal fmt_type_ii        : std_logic_vector( 1 downto 0);

signal link0_map_i        : std_logic_vector( 3 downto 0);
signal link1_map_i        : std_logic_vector( 3 downto 0);
signal link2_map_i        : std_logic_vector( 3 downto 0);
signal link3_map_i        : std_logic_vector( 3 downto 0);
signal setup_link_map_i   : std_logic;

signal boc_scan_ctrl     : std_logic_vector( 1 downto 0);
signal boc_scan_gcount   : std_logic_vector(15 downto 0); --
signal boc_scan_count    : unsigned(15 downto 0); --
type   boc_scan_mcount_array is array (0 to 3) of unsigned(15 downto 0); --
signal boc_scan_mcount  : boc_scan_mcount_array; --
type   opto_scan_mcount_array is array (0 to 3) of unsigned(31 downto 0); --
signal opto_scan_mcount : opto_scan_mcount_array;

signal simu_config_1_int : std_logic_vector(31 downto 0);
signal simu_config_2_int : std_logic_vector(31 downto 0);

signal l1id_i      : std_logic_vector(23 downto 0);
signal bcr_count   : std_logic_vector(15 downto 0);
signal ecr_count   : std_logic_vector(15 downto 0);
signal ecr_count_r : std_logic_vector(15 downto 0);
signal evt_count_r : std_logic;

signal dbg_count_ctrl_i : std_logic_vector( 1 downto 0);

------------------------------------------------------------------------------
-- PROCESS DECLARATION
-------------------------------------------------------------------------------
begin

simu_config_1 <= simu_config_1_int;
simu_config_2 <= simu_config_2_int;

-------------------------------------------------------------------------------
-- Process to Enable Read/Write to Router Registers
-------------------------------------------------------------------------------
rw_registers : process (
  clk40_in, 
  rst_n_in,
  frm_bus_addr_in,
  frm_bus_rnw_in,
  frm_bus_strb_in
  )
constant code_version : std_logic_vector(7 downto 0) := X"44";   -- v44
constant xper_version : std_logic_vector(3 downto 0) := X"4" ;    -- v4
constant board_rev    : std_logic_vector(4 downto 0) := "01111"; -- RevF
constant rod_type     : std_logic := '1';                        -- Pixel
variable setup_delay  : unsigned(2 downto 0);
begin
  if (rst_n_in = '0') then
    frm_bus_data_out   <= (others => '0');
    edge_mode_i        <= (others => '0');
    ecr_link_flush_i   <= (others => '0');
----*********************************************************
--            Synth/Build values
    link_enabled_i     <= (others => '0');
    config_mode_i      <= (others => '0');
    num_accepts_i      <= (others => '0');
    timeout_limit_i    <= X"00000800"; -- > 51.6uS --
    simu_config_1_int  <= (others => '0');
    simu_config_2_int  <= (others => '0');
    overflow_limit_i   <= X"0800";
----*********************************************************
--            Simulation values
-- pragma translate_off
    link_enabled_i     <= X"001";
--    config_mode_i      <= X"00F";
    num_accepts_i      <= X"7777";
    timeout_limit_i    <= X"00000080"; -- 80 ==3.2us
    simu_config_1_int  <= (others => '0'); --X"00000071"; -- MCC=0, NA=8, Enabled
    simu_config_2_int  <= (others => '0'); --X"04000000"; -- Num Hits(31:24)=0
    overflow_limit_i   <= X"0100";
-- pragma translate_on
----*********************************************************
    ht_limit_i         <= X"0700";
    rod_busy_limit_i   <= X"0780";  -- p3Bf value 0x600
    en_mb_fifo_rd_i    <= '0';
    clr_to_status_i    <= '0';
    clr_dovr_status_i  <= '0';
    clr_nnr_status_i   <= '0';
    clr_htl_status_i   <= '0';
    clr_rbl_status_i   <= '0';
    en_link_fifo_rd_i  <= '0';
    en_link_fifo_wr_i  <= '0';
    link_fifo_select_i <= (others => '0');
    mb_rst_n_i         <= '1';
    rod_type_out       <= '0';
    setup_link_map_i   <= '1';
    setup_delay        := (others => '0');
    dlink_out_mux_i    <= (others => '1');
    boc_scan_ctrl      <= (others => '0');
    boc_scan_gcount    <= (others => '0');
    ecr_count_r        <= (others => '0');
    evt_count_r        <= '0';
  elsif (clk40_in'event AND clk40_in = '1') then
  -- Set Formatter ID and map input links to match the BOC
    if (setup_link_map_i = '1') then
      if (fmt_number_in = "10") then
      -- Formatters 2 & 6
        link0_map_i <= X"8"; -- 0x8
        link1_map_i <= X"9"; -- 0x9
        link2_map_i <= X"A"; -- 0xA
        link3_map_i <= X"B"; -- 0xB
      else
      -- Formatters 0,1,3,4,5 & 7
        link0_map_i <= X"0"; -- 0x0
        link1_map_i <= X"1"; -- 0x1
        link2_map_i <= X"2"; -- 0x2
        link3_map_i <= X"3"; -- 0x3
      end if;
      if (setup_delay = 7) then  -- 200ns delay before setting values
        setup_link_map_i <= '0';
        setup_delay := (others => '0');
      else
        setup_delay := setup_delay + 1 ;
      end if;
    end if;

 --Read registers
    frm_bus_data_out <= (others => '0');
    if (frm_bus_rnw_in = '1' AND frm_bus_strb_in = '1') then
      case frm_bus_addr_in is
        when X"00" =>
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out(11 downto 0) <= link_enabled_i;
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= (others => '0');
          end if;
        when X"01" =>
	        if (frm_bus_hwsb_in = '0') then
	          frm_bus_data_out(11 downto 0) <= ecr_link_flush_i;
          elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= (others => '0');
          end if;
        when X"02" =>
	        if (frm_bus_hwsb_in = '0') then
	          frm_bus_data_out(11 downto 0) <= config_mode_i;
          elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= (others => '0');
          end if;
        when X"03" =>
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out(11 downto 0) <= edge_mode_i;
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= (others => '0');
          end if;
        when X"04" =>
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= timeout_limit_i(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= timeout_limit_i(31 downto 16);
          end if;
        when X"05" =>
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= overflow_limit_i;
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= (others => '0');
          end if;
        when X"06" =>
	        if (frm_bus_hwsb_in = '0') then
	           frm_bus_data_out <= ht_limit_i;
          elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= (others => '0');
          end if;
        when X"07" => 
	        if (frm_bus_hwsb_in = '0') then
	          frm_bus_data_out <= rod_busy_limit_i;
          elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= (others => '0');
          end if;
        when X"08" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= num_accepts_i;
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= num_accepts_in;
          end if;
        when X"09" =>
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out( 1 downto 0) <= fmt_type_ii;
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= link3_map_i & link2_map_i & link1_map_i & link0_map_i;
          end if;
        when X"0C" => -- L1A count
          if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= l1id_i(15 downto 0);
          else
            frm_bus_data_out <= X"00" & l1id_i(23 downto 16);
          end if;
        when X"0D" => -- ECR & BCR count
          if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= ecr_count;
          else
            frm_bus_data_out <= bcr_count;
          end if;
        when X"0E" =>
	        if (frm_bus_hwsb_in = '0') then
	          frm_bus_data_out( 3 downto 0) <= dlink_out_mux_i;
          elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= (others => '0');
          end if;
        when X"0F" => -- diagnostics;
	        if (frm_bus_hwsb_in = '0') then
	          frm_bus_data_out(0) <= en_mb_fifo_rd_i;
	          frm_bus_data_out(1) <= en_link_fifo_rd_i;
	          frm_bus_data_out(2) <= en_link_fifo_wr_i;
	          frm_bus_data_out(3) <= '0';
	          frm_bus_data_out(7 downto 4) <= link_fifo_select_i;
          elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= (others => '0');
          end if;

        when X"10" =>
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out(12 downto 0) <= occupancy_count_link0;
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= (others => '0');
          end if;
        when X"11" =>
	        if (frm_bus_hwsb_in = '0') then
	          frm_bus_data_out(12 downto 0) <= occupancy_count_link1;
          elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= (others => '0');
          end if;
        when X"12" =>
	        if (frm_bus_hwsb_in = '0') then
	          frm_bus_data_out(12 downto 0) <= occupancy_count_link2;
          elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= (others => '0');
          end if;
        when X"13" =>
	        if (frm_bus_hwsb_in = '0') then
	          frm_bus_data_out(12 downto 0) <= occupancy_count_link3;
          elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= (others => '0');
          end if;
        when X"1C" =>
          clr_to_status_i <= '1';
	        if (frm_bus_hwsb_in = '0') then
	          frm_bus_data_out(11 downto 0) <= time_out_error_i;
          elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= (others => '0');
          end if;
        when X"1D" =>
          clr_dovr_status_i <= '1';
	        if (frm_bus_hwsb_in = '0') then
	          frm_bus_data_out(11 downto 0) <= dovr_error_i;
          elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= (others => '0');
          end if;
        when X"1E" =>
          clr_htl_status_i <= '1';
	        if (frm_bus_hwsb_in = '0') then
	          frm_bus_data_out(11 downto 0) <= htl_error_i;
          elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= (others => '0');
          end if;
        when X"1F" =>
          clr_rbl_status_i <= '1';
	        if (frm_bus_hwsb_in = '0') then
	          frm_bus_data_out(11 downto 0) <= rbl_error_i;    --rod_busy_error;
          elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= (others => '0');
          end if;
        when X"20" =>
          clr_nnr_status_i <= '1';
	        if (frm_bus_hwsb_in = '0') then
	          frm_bus_data_out(11 downto 0) <= normal_not_raw_i;
          elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= (others => '0');
          end if;
        when X"21" => 
    	    if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= frm_dll_locked     &
                                master_not_slave   &
                                hold_output        &
                                chip_has_token_in  &
                                active_link_status &
                                mb_fifo_ff         &
                                mb_fifo_ef         &
                                mb_rst_n_i         &
                                trigger_count(4 downto 0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= "000000" & trigger_count;
          end if;
        when X"22" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= rod_type      & 
                                fmt_type_i(1) & 
                                fmt_type_i(0) &
                                board_rev     & 
                                code_version;
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= spare_pin_reg_in &
                                xper_version     &
                                fmt_type_in(1)   &
                                fmt_type_in(0)   &
                                fmt_number_in(1) &
                                fmt_number_in(0);
          end if;
        when X"23" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mode_bits(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= readout_status_in & "0000" & mode_bits(23 downto 16);
          end if;
        when X"24" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out ( 1 downto 0) <= boc_scan_ctrl(1 downto 0);
            frm_bus_data_out (15 downto 2) <= (others => '0');
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out  <= (others => '0');
          end if;
        when X"25" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= boc_scan_gcount;
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= std_logic_vector(boc_scan_count);
          end if;
        when X"26" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= std_logic_vector(boc_scan_mcount(0));
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= std_logic_vector(boc_scan_mcount(1));
          end if;
        when X"27" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= std_logic_vector(boc_scan_mcount(2));
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= std_logic_vector(boc_scan_mcount(3));
          end if;
        when X"28" => -- simulator configuration reg one
          if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= simu_config_1_int(15 downto  0);
          elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= simu_config_1_int(31 downto 16);
          end if;
        when X"29" => -- simulator configuration reg two 
          if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= simu_config_2_int(15 downto  0);
          elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= simu_config_2_int(31 downto 16);
          end if;
        when X"2C" =>  -- Mod0 Opto Counter
          if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= std_logic_vector(opto_scan_mcount(0)(15 downto  0));
          elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= std_logic_vector(opto_scan_mcount(0)(31 downto 16));
          end if;
        when X"2D" =>  -- Mod1 Opto Counter
          if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= std_logic_vector(opto_scan_mcount(1)(15 downto  0));
          elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= std_logic_vector(opto_scan_mcount(1)(31 downto 16));
          end if;
        when X"2E" =>  -- Mod2 Opto Counter
          if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= std_logic_vector(opto_scan_mcount(2)(15 downto  0));
          elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= std_logic_vector(opto_scan_mcount(2)(31 downto 16));
          end if;
        when X"2F" =>  -- Mod3 Opto Counter
          if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= std_logic_vector(opto_scan_mcount(3)(15 downto  0));
          elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= std_logic_vector(opto_scan_mcount(3)(31 downto 16));
          end if;


        when X"3C" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= link_fifo_data_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= link_fifo_data_in(31 downto 16);
          end if;
        when X"3F" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= "00000000000000" & dbg_count_ctrl_i;
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= (others => '0');
          end if;
        when X"40" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod0_resync_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod0_resync_count_in(31 downto 16);
          end if;
        when X"41" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod0_hitword_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod0_hitword_count_in(31 downto 16);
          end if;
        when X"42" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod0_event_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod0_event_count_in(31 downto 16);
          end if;
        when X"43" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod0_error_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod0_error_count_in(31 downto 16);
          end if;
        when X"44" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod0_dproc_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod0_dproc_count_in(31 downto 16);
          end if;
        when X"45" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod0_clk_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod0_clk_count_in(31 downto 16);
          end if;
        when X"48" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod1_resync_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod1_resync_count_in(31 downto 16);
          end if;
        when X"49" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod1_hitword_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod1_hitword_count_in(31 downto 16);
          end if;
        when X"4A" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod1_event_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod1_event_count_in(31 downto 16);
          end if;
        when X"4B" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod1_error_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod1_error_count_in(31 downto 16);
          end if;
        when X"4C" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod1_dproc_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod1_dproc_count_in(31 downto 16);
          end if;
        when X"4D" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod1_clk_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod1_clk_count_in(31 downto 16);
          end if;
        when X"50" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod2_resync_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod2_resync_count_in(31 downto 16);
          end if;
        when X"51" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod2_hitword_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod2_hitword_count_in(31 downto 16);
          end if;
        when X"52" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod2_event_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod2_event_count_in(31 downto 16);
          end if;
        when X"53" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod2_error_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod2_error_count_in(31 downto 16);
          end if;
        when X"54" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod2_dproc_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod2_dproc_count_in(31 downto 16);
          end if;
        when X"55" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod2_clk_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod2_clk_count_in(31 downto 16);
          end if;
        when X"58" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod3_resync_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod3_resync_count_in(31 downto 16);
          end if;
        when X"59" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod3_hitword_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod3_hitword_count_in(31 downto 16);
          end if;
        when X"5A" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod3_event_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod3_event_count_in(31 downto 16);
          end if;
        when X"5B" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod3_error_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod3_error_count_in(31 downto 16);
          end if;
        when X"5C" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod3_dproc_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod3_dproc_count_in(31 downto 16);
          end if;
        when X"5D" => 
	        if (frm_bus_hwsb_in = '0') then
            frm_bus_data_out <= mod3_clk_count_in(15 downto  0);
	        elsif (frm_bus_hwsb_in = '1') then
            frm_bus_data_out <= mod3_clk_count_in(31 downto 16);
          end if;
          
        when others => frm_bus_data_out <= (others => '0');
      end case;
 --Write registers
    elsif (frm_bus_rnw_in = '0' AND frm_bus_strb_in = '1') then
      case frm_bus_addr_in is
        when X"00" =>
          if (frm_bus_hwsb_in = '0' AND frm_bus_ds_in = '1') then
            case fmt_type_i is
              when "01" =>  -- 80MHz Decoder
                link_enabled_i(0)           <= frm_bus_data_in(0);
                link_enabled_i(1)           <= frm_bus_data_in(1);
                link_enabled_i(11 downto 2) <= (others => '0');
              when "10" =>  -- 160MHz decoder
                link_enabled_i(0)           <= frm_bus_data_in(0);
                link_enabled_i(11 downto 1) <= (others => '0');
              when others =>
                link_enabled_i <= frm_bus_data_in(11 downto 0);
            end case;

--        elsif (frm_bus_hwsb_in = '1' AND frm_bus_ds_in = '1') then
          end if;
        when X"01" =>  -- SCT Config Expanded Mode
          if (frm_bus_hwsb_in = '0' AND frm_bus_ds_in = '1') then
            ecr_link_flush_i <= frm_bus_data_in(11 downto 0);
          end if;
        when X"02" =>  -- Raw Data Capture
          if (frm_bus_hwsb_in = '0' AND frm_bus_ds_in = '1') then
            config_mode_i <= frm_bus_data_in(11 downto 0);
          end if;
        when X"03" =>  -- SCT Config
          if (frm_bus_hwsb_in = '0' AND frm_bus_ds_in = '1') then
            edge_mode_i <= frm_bus_data_in(11 downto 0);
          end if;
        when X"04" =>
          if (frm_bus_hwsb_in = '0' AND frm_bus_ds_in = '1') then
            timeout_limit_i(15 downto  0) <= frm_bus_data_in;
          elsif (frm_bus_hwsb_in = '1' AND frm_bus_ds_in = '1') then
            timeout_limit_i(31 downto 16) <= frm_bus_data_in;
          end if;
        when X"05" =>
          if (frm_bus_hwsb_in = '0' AND frm_bus_ds_in = '1') then
            overflow_limit_i <= frm_bus_data_in;
          end if;
        when X"06" =>
          if (frm_bus_hwsb_in = '0' AND frm_bus_ds_in = '1') then
            ht_limit_i <= frm_bus_data_in;
          end if;
        when X"07" =>
          if (frm_bus_hwsb_in = '0' AND frm_bus_ds_in = '1') then
            rod_busy_limit_i <= frm_bus_data_in;
          end if;
        when X"08" =>
          if (frm_bus_hwsb_in = '0' AND frm_bus_ds_in = '1') then
            num_accepts_i <= frm_bus_data_in;
          end if;
        when X"09" =>
          if (frm_bus_hwsb_in = '0' AND frm_bus_ds_in = '1') then
            fmt_type_ii <= frm_bus_data_in( 1 downto 0);
          elsif (frm_bus_hwsb_in = '1' AND frm_bus_ds_in = '1') then
            link0_map_i <= frm_bus_data_in( 3 downto  0);
            link1_map_i <= frm_bus_data_in( 7 downto  4);
            link2_map_i <= frm_bus_data_in(11 downto  8);
            link3_map_i <= frm_bus_data_in(15 downto 12);
          end if;
        when X"0D" => -- ECR & BCR count
          if (frm_bus_hwsb_in = '0') then
            ecr_count_r <= frm_bus_data_in(15 downto  0);
          else
            evt_count_r <= frm_bus_data_in(0);
          end if;
        when X"0E" =>
          if (frm_bus_hwsb_in = '0' AND frm_bus_ds_in = '1') then
            dlink_out_mux_i <= frm_bus_data_in( 3 downto 0);
          end if;
        when X"0F" =>
          if (frm_bus_hwsb_in = '0' AND frm_bus_ds_in = '1') then
             en_mb_fifo_rd_i    <= frm_bus_data_in(0);
             en_link_fifo_rd_i  <= frm_bus_data_in(1);
             en_link_fifo_wr_i  <= frm_bus_data_in(2);
             link_fifo_select_i <= frm_bus_data_in(7 downto 4);
          end if;
        when X"24" => 
	        if (frm_bus_hwsb_in = '0' AND frm_bus_ds_in = '1') then
            boc_scan_ctrl(1 downto 0) <= frm_bus_data_in(1 downto 0);
          end if;
        when X"25" => 
	        if (frm_bus_hwsb_in = '0' AND frm_bus_ds_in = '1') then
            boc_scan_gcount <= frm_bus_data_in;
          end if;
--      when X"26" =>  READ ONLY
--        boc_scan_m0count;
--        boc_scan_m1count;
--      when X"27" => 
--	    boc_scan_m2count;
--        boc_scan_m3count;
        when X"28" => -- simulator configuration reg one
          if (frm_bus_hwsb_in = '0') then
            simu_config_1_int(15 downto  0) <= frm_bus_data_in;
          elsif (frm_bus_hwsb_in = '1') then
            simu_config_1_int(31 downto 16) <= frm_bus_data_in;
          end if;
        when X"29" => -- simulator configuration reg two
          if (frm_bus_hwsb_in = '0') then
            simu_config_2_int(15 downto  0) <= frm_bus_data_in;
          elsif (frm_bus_hwsb_in = '1') then
            simu_config_2_int(31 downto 16) <= frm_bus_data_in;
          end if;
        when X"3C" => 
          if (frm_bus_hwsb_in = '0' AND frm_bus_ds_in = '1') then
            link_fifo_data_out(15 downto  0) <= frm_bus_data_in;
          elsif (frm_bus_hwsb_in = '1' AND frm_bus_ds_in = '1') then
            link_fifo_data_out(31 downto 16) <= frm_bus_data_in;
          end if;
        when X"3F" => 
          if (frm_bus_hwsb_in = '0' AND frm_bus_ds_in = '1') then
            dbg_count_ctrl_i <= frm_bus_data_in(1 downto 0);
          end if;

        when others => -- NOP
      end case;
    else
      link_enabled          <= link_enabled_i;
      ecr_link_flush        <= ecr_link_flush_i;
      config_mode           <= config_mode_i;
      edge_mode             <= edge_mode_i;
      timeout_limit         <= unsigned(timeout_limit_i);
      overflow_limit        <= unsigned(overflow_limit_i);
      header_trailer_limit  <= unsigned(ht_limit_i);
      rod_busy_limit        <= unsigned(rod_busy_limit_i);
      num_accepts_out       <= num_accepts_i;
      clr_to_status_i       <= '0';
      clr_dovr_status_i     <= '0';
      clr_nnr_status_i      <= '0';
      clr_htl_status_i      <= '0';
      clr_rbl_status_i      <= '0';
      data_link_out_mux_sel <= dlink_out_mux_i;
      en_mb_fifo_rd_i       <= '0';
      en_link_fifo_rd_i     <= '0';
      en_link_fifo_wr_i     <= '0';
      link_fifo_select      <= link_fifo_select_i; 
      fmt_type_out          <= fmt_type_i;
      link0_map_out         <= link0_map_i;
      link1_map_out         <= link1_map_i;
      link2_map_out         <= link2_map_i;
      link3_map_out         <= link3_map_i;
      evt_count_r           <= '0';
      dbg_count_ctrl_out    <= dbg_count_ctrl_i;
      case fmt_type_in is
        when "00"   => fmt_type_i <= fmt_type_ii; -- default to 40Mb and allow Software control
        when "11"   => fmt_type_i <= "00";        -- 40Mb
        when others => fmt_type_i <= fmt_type_in; -- 80Mb & 160Mb
      end case;
    end if;
    mb_rst_n_i   <= mb_rst_n_in;
    rod_type_out <= rod_type;
  end if;
end process rw_registers;

event_id_counters : process (
  rst_n_in, 
  clk40_in
  )
begin                      
  if (rst_n_in = '0') then
    l1id_i <= (others => '0'); -- same as FE Module
    bcr_count <= (others => '0'); 
    ecr_count <= (others => '0');
--    pres_l1id_out <= (others => '0');
    last_l1id_out <= (others => '0');
  elsif (clk40_in'event and clk40_in = '1') then
    if (evt_count_r = '1') then
      bcr_count <= (others => '0'); 
      ecr_count <= ecr_count_r;
    elsif (ecr_in = '1') then
      last_l1id_out <= l1id_i;
      l1id_i <= (others => '0');
      ecr_count <= ecr_count + '1';
    elsif (l1a_in = '1') then
      l1id_i <= l1id_i + '1';
    end if;
 
    if (bcr_in = '1') then
      bcr_count <= bcr_count + '1';
    end if;
  end if;  
end process;

pres_l1id_out <= l1id_i(3 downto 0);

error_status : process (
  rst_n_in, 
  clk40_in
  )
begin                      
  if (rst_n_in = '0') then
    time_out_error_i  <= (others => '0');
    dovr_error_i      <= (others => '0');
    htl_error_i       <= (others => '0');
    rbl_error_i       <= (others => '0');
    normal_not_raw_i  <= (others => '0');
    ht_limit_en_n_out <= '1';
    rb_limit_en_n_out <= '1';
    do_error_out      <= '1';
  elsif (clk40_in'event and clk40_in = '1') then
    if (clr_to_reg_i = '1') then
      time_out_error_i <= time_out_error;
    else
      time_out_error_i <= (time_out_error_i OR time_out_error) AND link_enabled_i;
    end if;

    if (clr_dovr_reg_i = '1') then
      dovr_error_i <= data_overflow_error;
    else
      dovr_error_i <= (dovr_error_i OR data_overflow_error) AND link_enabled_i;
    end if;

    if (clr_nnr_reg_i = '1') then
      normal_not_raw_i <= normal_not_raw;
    else
      normal_not_raw_i <= (normal_not_raw_i OR normal_not_raw) AND link_enabled_i;
    end if;

    if (clr_htl_reg_i = '1') then
      htl_error_i <= header_trailer_error AND link_enabled_i;
    else
      htl_error_i <= (htl_error_i OR header_trailer_error) AND link_enabled_i;
    end if;

    if (htl_error_i = "000000000000") then
      ht_limit_en_n_out <= '1';
    else
      ht_limit_en_n_out <= '0';
    end if;

    if (clr_rbl_reg_i = '1') then
      rbl_error_i <= rod_busy_error AND link_enabled_i;                                                      
    else
      rbl_error_i <= (rbl_error_i OR rod_busy_error) AND link_enabled_i; 
    end if;
 
    if (rod_busy_error(3 downto 0) = "0000") then                                                         
      rb_limit_en_n_out <= '1';
    else
      rb_limit_en_n_out <= '0';
    end if;

    if (dovr_error_i = "000000000000") then
      do_error_out <= '0';
    else
      do_error_out <= '1';
    end if;
  end if;  
end process error_status;     

boc_scan : process( 
  rst_n_in, 
  clk40_in
  )
begin                      
  if (rst_n_in = '0') then
    boc_scan_count <= (others => '0');
    for i in 0 to 3 loop 
      boc_scan_mcount(i)  <= (others => '0');
      opto_scan_mcount(i) <= (others => '0');
    end loop;
  elsif (clk40_in'event and clk40_in = '1') then
    if (boc_scan_ctrl(0) = '0') then
      for i in 0 to 3 loop 
        boc_scan_mcount(i) <= (others => '0');
        opto_scan_mcount(i) <= (others => '0');
      end loop;
      boc_scan_count <= unsigned(boc_scan_gcount);  -- load value only in reset
    elsif (boc_scan_ctrl = "01" AND boc_scan_count > 0) then
      boc_scan_count <= boc_scan_count - 1 ;
      for i in 0 to 3 loop 
        boc_scan_mcount(i) <= boc_scan_mcount(i) + boc_scan_data_in(i);
      end loop;
    elsif (boc_scan_ctrl = "11") then
      for i in 0 to 3 loop 
        boc_scan_mcount(i) <= boc_scan_mcount(i) + boc_scan_data_in(i);
      end loop;
      for i in 0 to 3 loop 
        if (boc_scan_mcount(i) = 65535 AND boc_scan_data_in(i) = '1') then
          opto_scan_mcount(i) <= opto_scan_mcount(i) + '1';
        end if;
      end loop;
    end if;
  end if;  
end process boc_scan;     

-------------------------------------------------------------------------------
-- Logic to generate Access Acknowledge
-------------------------------------------------------------------------------
access_acknowledge : process (
  rst_n_in, 
  clk40_in, 
  frm_bus_strb_in
  )
begin                      
  if (rst_n_in = '0') then
    frm_bus_ack_out <= '0';
    clr_to_reg_i    <= '0';
    clr_dovr_reg_i  <= '0';
    clr_nnr_reg_i   <= '0';
    clr_htl_reg_i   <= '0';
    clr_rbl_reg_i   <= '0';
    mb_fifo_rd      <= '0';
    link_fifo_rd    <= '0';
    link_fifo_wr    <= '0';
    acknowledge_cycle_state <= idle;
  elsif (clk40_in'event and clk40_in = '1') then
    case acknowledge_cycle_state is
      when idle => 
        frm_bus_ack_out <= '0';
        clr_to_reg_i    <= '0';
        clr_dovr_reg_i  <= '0';
        clr_nnr_reg_i   <= '0';
        clr_htl_reg_i   <= '0';
        clr_rbl_reg_i   <= '0';
        if (frm_bus_strb_in = '1') then 
          mb_fifo_rd   <= en_mb_fifo_rd_i;
          link_fifo_rd <= en_link_fifo_rd_i;
          link_fifo_wr <= en_link_fifo_wr_i;
          acknowledge_cycle_state <= acknowledge;
        end if;
       
      when acknowledge => 
        frm_bus_ack_out <= '1';
        link_fifo_rd    <= '0';
        link_fifo_wr    <= '0';
        mb_fifo_rd      <= '0';
        if (frm_bus_strb_in = '0') then 
          acknowledge_cycle_state <= idle;
          clr_to_reg_i   <= clr_to_status_i;
          clr_dovr_reg_i <= clr_dovr_status_i;
          clr_nnr_reg_i  <= clr_nnr_status_i;
          clr_htl_reg_i  <= clr_htl_status_i;
          clr_rbl_reg_i  <= clr_rbl_status_i;
        end if;

      when others =>
        frm_bus_ack_out <= '0';
        mb_fifo_rd      <= '0';
        acknowledge_cycle_state <= idle;
    end case;
  end if;  
end process access_acknowledge;     

end rtl; -- code of register_block
