----------------------------------------------------------------------------
-- LBL - WISCONSIN
-- ATLAS ROD ELECTRONICS
----------------------------------------------------------------------------
-- Filename:
--    quad_link_formatter.vhd
-- Description:
--    ROD Front End Decoder
--       
--
----------------------------------------------------------------------------

library IEEE;
use IEEE.STD_LOGIC_1164.all;
use IEEE.std_logic_arith.all;

----------------------------------------------------------------------------
--PORT DECLARATION
----------------------------------------------------------------------------

entity quad_link_formatter is
  port(
    clk_in                 : in  std_logic; -- 40 MHz clock
    rst_n_in               : in  std_logic; -- Powerup global reset
 -- enable link formatter activity
    link_enable_in         : in  std_logic_vector( 3 DOWNTO 0);
    ecr_link_flush         : in  std_logic_vector( 3 DOWNTO 0);
    decode_type_in         : in  std_logic_vector( 1 DOWNTO 0);
    config_mode_in         : in  std_logic_vector( 3 DOWNTO 0); -- 1 = raw
                                                                -- 0 = normal
    half_clk_err_mask_in   : in  std_logic_vector( 3 DOWNTO 0);                                                                
    link_data_in           : in  std_logic_vector( 3 DOWNTO 0); -- front end serial data
    ht_limit_in            : in  unsigned(15 DOWNTO 0); -- Header/Trailer Limit value
    rod_busy_limit_in      : in  unsigned(15 DOWNTO 0); -- ROD Busy Limit value
    num_accepts_in         : in  std_logic_vector(15 DOWNTO 0);
    link_to_error_in       : in  std_logic_vector( 3 DOWNTO 0);  -- Link Time out
    fifo_ren_in            : in  std_logic_vector( 3 DOWNTO 0);  -- fifo REN
    fifo_pause_in          : in  std_logic_vector( 3 DOWNTO 0);  -- fifo pause
    normal_not_raw_out     : out std_logic_vector( 3 DOWNTO 0);  -- 1 = normal data mode
                                                                 -- 0 = raw data mode 
    decoder_wr_trailer_out : out std_logic_vector( 3 DOWNTO 0); -- trailer being written to FIFO
    ht_limit_out           : out std_logic_vector( 3 DOWNTO 0);
    rod_busy_limit_out     : out std_logic_vector( 3 DOWNTO 0);
    fifo_ef_out            : out std_logic_vector( 3 DOWNTO 0);
    fifo_ff_out            : out std_logic_vector( 3 DOWNTO 0);
--  trailer_detect_out     : out std_logic_vector(11 DOWNTO 0);
    occ_count_out          : out std_logic_vector(51 DOWNTO 0);
    fifo0_data_out         : out std_logic_vector(31 DOWNTO 0); -- Output data word
    fifo1_data_out         : out std_logic_vector(31 DOWNTO 0); -- Output data word
    fifo2_data_out         : out std_logic_vector(31 DOWNTO 0); -- Output data word
    fifo3_data_out         : out std_logic_vector(31 DOWNTO 0); -- Output data word
    data_valid_out         : out std_logic_vector( 3 DOWNTO 0);
    eoe_word_out           : out std_logic_vector( 3 DOWNTO 0);
    num_accepts_out        : out std_logic_vector(15 downto 0);
    ecr_in                 : in  std_logic;
    tim_l1id_in            : in  std_logic_vector( 3 DOWNTO 0);
    last_l1id_in           : in  std_logic_vector(23 DOWNTO 0);
    dbg_count_ctrl_in      : in  std_logic_vector( 1 downto 0);
    mod0_resync_count_out  : out std_logic_vector(31 DOWNTO 0);
    mod0_hitword_count_out : out std_logic_vector(31 DOWNTO 0);
    mod0_event_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod0_error_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod0_dproc_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod0_clk_count_out     : out std_logic_vector(31 DOWNTO 0);
    mod1_resync_count_out  : out std_logic_vector(31 DOWNTO 0);
    mod1_hitword_count_out : out std_logic_vector(31 DOWNTO 0);
    mod1_event_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod1_error_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod1_dproc_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod1_clk_count_out     : out std_logic_vector(31 DOWNTO 0);
    mod2_resync_count_out  : out std_logic_vector(31 DOWNTO 0);
    mod2_hitword_count_out : out std_logic_vector(31 DOWNTO 0);
    mod2_event_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod2_error_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod2_dproc_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod2_clk_count_out     : out std_logic_vector(31 DOWNTO 0);
    mod3_resync_count_out  : out std_logic_vector(31 DOWNTO 0);
    mod3_hitword_count_out : out std_logic_vector(31 DOWNTO 0);
    mod3_event_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod3_error_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod3_dproc_count_out   : out std_logic_vector(31 DOWNTO 0);
    mod3_clk_count_out     : out std_logic_vector(31 DOWNTO 0)
    );
end quad_link_formatter ; 

architecture rtl of quad_link_formatter is

----------------------------------------------------------------------------
--SIGNAL DECLARATION
----------------------------------------------------------------------------

type   fifo_data_array is array (0 to 3) of std_logic_vector(31 downto 0); --
signal fifo_data_i     : fifo_data_array; --
signal fifo_data_o     : fifo_data_array; --
signal fifo_data_d40_i : fifo_data_array; --

type   fifo_data_d80_array is array (0 to 1) of std_logic_vector(31 downto 0); --
signal fifo_data_d80_i : fifo_data_d80_array; --

signal fifo_data_d160_i : std_logic_vector(31 downto 0); --

type   fifo_occ_array is array (0 to 3) of unsigned(10 downto 0); --
signal occ_count_i : fifo_occ_array; --

type   na_array is array (0 to 3) of std_logic_vector(3 downto 0); --
signal num_accepts_o : na_array; --

type   debug_counter_array is array (0 to 3) of std_logic_vector(31 downto 0); --
signal resync_count_i  : debug_counter_array; --
signal hitword_count_i : debug_counter_array; --
signal event_count_i   : debug_counter_array; --
signal error_count_i   : debug_counter_array; --
signal dproc_count_i   : debug_counter_array; --
signal clk_count_i     : debug_counter_array; --

signal link_data_i  : std_logic_vector(3 DOWNTO 0);
signal ht_limit_i   : std_logic_vector(3 DOWNTO 0);
signal rb_limit_i   : std_logic_vector(3 DOWNTO 0);
signal fifo_wen_i   : std_logic_vector(3 DOWNTO 0);
signal data_valid_i : std_logic_vector(3 DOWNTO 0);
signal fifo_ef_i    : std_logic_vector(3 DOWNTO 0);
signal fifo_ff_i    : std_logic_vector(3 DOWNTO 0);
signal ecr_i        : std_logic_vector(3 DOWNTO 0);

signal dec40_nnr_i  : std_logic_vector(3 DOWNTO 0);
signal dec80_nnr_i  : std_logic_vector(1 DOWNTO 0);
signal dec160_nnr_i : std_logic;

signal dec40_fifo_wen_i  : std_logic_vector(3 DOWNTO 0);
signal dec80_fifo_wen_i  : std_logic_vector(1 DOWNTO 0);
signal dec160_fifo_wen_i : std_logic;

signal dec40_wr_trailer_i  : std_logic_vector(3 DOWNTO 0);
signal dec80_wr_trailer_i  : std_logic_vector(1 DOWNTO 0);
signal dec160_wr_trailer_i : std_logic;
 
signal decoder_dproc_en     : std_logic_vector(3 DOWNTO 0);
signal d40_decoder_in_idle  : std_logic_vector(3 DOWNTO 0);
signal d80_decoder_in_idle  : std_logic_vector(1 DOWNTO 0);
signal d160_decoder_in_idle : std_logic;
 
----------------------------------------------------------------------------
--COMPONENT DECLARATION
----------------------------------------------------------------------------

component link_formatter_d40
  port (
    clk_in                     : in  std_logic;
    rst_n_in                   : in  std_logic;
    almost_full_in             : in  std_logic;
    enable_in                  : in  std_logic;
    config_mode_in             : in  std_logic;
    half_clk_err_mask_in       : in  std_logic;
    link_in                    : in  std_logic;
    normal_not_raw_out         : out std_logic;
    data_strobe_out            : out std_logic;
    decoder_writes_trailer_out : out std_logic;
    decoded_data_out           : out std_logic_vector(31 downto 0);
    decoder_state_in_idle      : out std_logic
    );
end component ; 

component link_formatter_d80
  port (
    clk_in                     : in  std_logic;
    rst_n_in                   : in  std_logic;
    almost_full_in             : in  std_logic;
    enable_in                  : in  std_logic;
    config_mode_in             : in  std_logic;
    half_clk_err_mask_in       : in  std_logic;
    link_in                    : in  std_logic_vector(1 downto 0);
    normal_not_raw_out         : out std_logic;
    data_strobe_out            : out std_logic;
    decoder_writes_trailer_out : out std_logic;
    decoded_data_out           : out std_logic_vector(31 downto 0);
    decoder_state_in_idle      : out std_logic
    );
end component ; 

component link_formatter_d160
  port (
    clk_in                     : in  std_logic;
    rst_n_in                   : in  std_logic;
    almost_full_in             : in  std_logic;
    enable_in                  : in  std_logic;
    config_mode_in             : in  std_logic;
    half_clk_err_mask_in       : in  std_logic;
    link_in                    : in  std_logic_vector(3 downto 0);
    normal_not_raw_out         : out std_logic;
    data_strobe_out            : out std_logic;
    decoder_writes_trailer_out : out std_logic;
    decoded_data_out           : out std_logic_vector(31 downto 0);
    decoder_state_in_idle      : out std_logic
    );
end component ; 

component fifo_2048x32
  port(
    clk_in            : in  std_logic;
    rst_n_in          : in  std_logic;
    link_enable_in    : in  std_logic;
    config_mode_in    : in  std_logic;
    num_accepts_in    : in  std_logic_vector( 3 downto 0); --
    to_error_in       : in  std_logic;
    wren_in           : in  std_logic;
    rden_in           : in  std_logic;
    fifo_pause_in     : in  std_logic;
    fifo_in           : in  std_logic_vector(31 downto 0); --
    fifo_out          : out std_logic_vector(31 downto 0); --
    data_valid_out    : out std_logic; --
    fifo_ef_out       : out std_logic;
    fifo_ff_out       : out std_logic;
    occupancy_out     : out unsigned(10 downto 0);
    fifo_eoe_word_out : out std_logic;
    num_accepts_out   : out std_logic_vector( 3 downto 0);
    ecr_in            : in  std_logic ;
    tim_l1id_in       : in  std_logic_vector( 3 DOWNTO 0);
    last_l1id_in      : in  std_logic_vector(23 DOWNTO 0);
    dbg_count_ctrl_in : in  std_logic_vector( 1 downto 0);
    dproc_count_en_in : in  std_logic;
    ecr_count_en_in   : in  std_logic;
    resync_count_out  : out std_logic_vector(31 DOWNTO 0);
    hitword_count_out : out std_logic_vector(31 DOWNTO 0);
    event_count_out   : out std_logic_vector(31 DOWNTO 0);
    error_count_out   : out std_logic_vector(31 DOWNTO 0);
    dproc_count_out   : out std_logic_vector(31 DOWNTO 0);
    clk_count_out     : out std_logic_vector(31 DOWNTO 0)
    );
end component;


BEGIN  --  Main Body of VHDL code
----------------------------------------------------------------------------
--COMPONENT INSTANTIATION
----------------------------------------------------------------------------

U1 : for i in 0 to 3 generate
  dec40 : link_formatter_d40
    port map(
      clk_in                     => clk_in,
      rst_n_in                   => rst_n_in,
      almost_full_in             => ht_limit_i(i),
      enable_in                  => link_enable_in(i),
      config_mode_in             => config_mode_in(i),
      half_clk_err_mask_in       => half_clk_err_mask_in(i),
      link_in                    => link_data_i(i),
      normal_not_raw_out         => dec40_nnr_i(i),
      data_strobe_out            => dec40_fifo_wen_i(i),
      decoder_writes_trailer_out => dec40_wr_trailer_i(i),
      decoded_data_out           => fifo_data_d40_i(i),
      decoder_state_in_idle      => d40_decoder_in_idle(i)
      );
end generate;

U2 : for i in 0 to 1 generate
  dec80 : link_formatter_d80
    port map(
      clk_in                     => clk_in,
      rst_n_in                   => rst_n_in,
      almost_full_in             => ht_limit_i(i),
      enable_in                  => link_enable_in(i),
      config_mode_in             => config_mode_in(i),
      half_clk_err_mask_in       => half_clk_err_mask_in(i),
      link_in                    => link_data_i((2*i)+1 downto 2*i),
      normal_not_raw_out         => dec80_nnr_i(i),
      data_strobe_out            => dec80_fifo_wen_i(i),
      decoder_writes_trailer_out => dec80_wr_trailer_i(i),
      decoded_data_out           => fifo_data_d80_i(i),
      decoder_state_in_idle      => d80_decoder_in_idle(i)
      );
end generate;

dec160 : link_formatter_d160
  port map(
    clk_in                     => clk_in,
    rst_n_in                   => rst_n_in,
    almost_full_in             => ht_limit_i(0),
    half_clk_err_mask_in       => half_clk_err_mask_in(0),
    enable_in                  => link_enable_in(0),
    config_mode_in             => config_mode_in(0),
    link_in                    => link_data_i,
    normal_not_raw_out         => dec160_nnr_i,
    data_strobe_out            => dec160_fifo_wen_i,
    decoder_writes_trailer_out => dec160_wr_trailer_i,
    decoded_data_out           => fifo_data_d160_i,
    decoder_state_in_idle      => d160_decoder_in_idle
    );

U4 : for i in 0 to 3 generate
  fmb : fifo_2048x32
    port map(
      clk_in            => clk_in,
      rst_n_in          => rst_n_in,
      link_enable_in    => link_enable_in(i),
      config_mode_in    => config_mode_in(i),
      num_accepts_in    => num_accepts_in(4*i+3 downto 4*i),
      to_error_in       => link_to_error_in(i),
      wren_in           => fifo_wen_i(i),
      rden_in           => fifo_ren_in(i),
      fifo_pause_in     => fifo_pause_in(i),
      fifo_in           => fifo_data_i(i),
      fifo_out          => fifo_data_o(i),
      data_valid_out    => data_valid_i(i),
      fifo_ef_out       => fifo_ef_i(i),
      fifo_ff_out       => fifo_ff_i(i),
      occupancy_out     => occ_count_i(i),
      fifo_eoe_word_out => eoe_word_out(i),
      num_accepts_out   => num_accepts_o(i),
      ecr_in            => ecr_i(i),
      tim_l1id_in       => tim_l1id_in,
      last_l1id_in      => last_l1id_in,
      dbg_count_ctrl_in => dbg_count_ctrl_in,
      ecr_count_en_in   => ecr_in,
      dproc_count_en_in => decoder_dproc_en(i),
      resync_count_out  => resync_count_i(i),
      hitword_count_out => hitword_count_i(i),
      event_count_out   => event_count_i(i),
      error_count_out   => error_count_i(i),
      dproc_count_out   => dproc_count_i(i),
      clk_count_out     => clk_count_i(i)
      );
end generate;


--------------------------------------------------------------------------
-- SIGNALS
--------------------------------------------------------------------------

occ_count_out <= "00" & std_logic_vector(occ_count_i(3)) &
                 "00" & std_logic_vector(occ_count_i(2)) &
                 "00" & std_logic_vector(occ_count_i(1)) &
                 "00" & std_logic_vector(occ_count_i(0));

ht_limit_out       <= ht_limit_i;
rod_busy_limit_out <= rb_limit_i;

fifo0_data_out <= fifo_data_o(0);
fifo1_data_out <= fifo_data_o(1);
fifo2_data_out <= fifo_data_o(2);
fifo3_data_out <= fifo_data_o(3);

data_valid_out <= data_valid_i;

fifo_ef_out <= fifo_ef_i;
fifo_ff_out <= fifo_ff_i;

num_accepts_out(15 downto 12) <= num_accepts_o(3);
num_accepts_out(11 downto  8) <= num_accepts_o(2);
num_accepts_out( 7 downto  4) <= num_accepts_o(1);
num_accepts_out( 3 downto  0) <= num_accepts_o(0);

decoder_dproc_en(0) <= NOT d40_decoder_in_idle(0) OR NOT d80_decoder_in_idle(0) OR NOT d160_decoder_in_idle;
decoder_dproc_en(1) <= NOT d40_decoder_in_idle(1) OR NOT d80_decoder_in_idle(1);
decoder_dproc_en(2) <= NOT d40_decoder_in_idle(2);
decoder_dproc_en(3) <= NOT d40_decoder_in_idle(3);

mod0_resync_count_out  <= resync_count_i(0);
mod0_hitword_count_out <= hitword_count_i(0);
mod0_event_count_out   <= event_count_i(0);
mod0_error_count_out   <= error_count_i(0);
mod0_dproc_count_out   <= dproc_count_i(0);
mod0_clk_count_out     <= clk_count_i(0);

mod1_resync_count_out  <= resync_count_i(1);
mod1_hitword_count_out <= hitword_count_i(1);
mod1_event_count_out   <= event_count_i(1);
mod1_error_count_out   <= error_count_i(1);
mod1_dproc_count_out   <= dproc_count_i(1);
mod1_clk_count_out     <= clk_count_i(1);

mod2_resync_count_out  <= resync_count_i(2);
mod2_hitword_count_out <= hitword_count_i(2);
mod2_event_count_out   <= event_count_i(2);
mod2_error_count_out   <= error_count_i(2);
mod2_dproc_count_out   <= dproc_count_i(2);
mod2_clk_count_out     <= clk_count_i(2);

mod3_resync_count_out  <= resync_count_i(3);
mod3_hitword_count_out <= hitword_count_i(3);
mod3_event_count_out   <= event_count_i(3);
mod3_error_count_out   <= error_count_i(3);
mod3_dproc_count_out   <= dproc_count_i(3);
mod3_clk_count_out     <= clk_count_i(3);

--------------------------------------------------------------------------
-- PROCESS DECLARATION
--------------------------------------------------------------------------
ecr_flush_map : process (
  rst_n_in,
  clk_in
  )
begin
  if (rst_n_in = '0') then
    ecr_i <= (others => '0');
  elsif (clk_in'event AND clk_in = '1') then
    ecr_i(0) <= ecr_in AND ecr_link_flush(0);
    ecr_i(1) <= ecr_in AND ecr_link_flush(1);
    ecr_i(2) <= ecr_in AND ecr_link_flush(2);
    ecr_i(3) <= ecr_in AND ecr_link_flush(3);
  end if;
end process;

link_input_map : process (
  rst_n_in,
  clk_in
  )
begin
  if (rst_n_in = '0') then
    link_data_i <= (others => '0');
  elsif (clk_in'event AND clk_in = '1') then
--  link_data_in(3) >> Input Link 3 : DTO2 fe
--  link_data_in(2) >> Input Link 2 : DTO2 re
--  link_data_in(1) >> Input Link 1 : DTO  fe
--  link_data_in(0) >> Input Link 0 : DTO  re
--
-- Input to Decoder Block :: link_data_i mapping ==
--  (3) >> DTO2_re
--  (2) >> DTO_re
--  (1) >> DTO2_fe
--  (0) >> DTO_fe

    case decode_type_in is
      when "01" =>  -- 80MHz NEW in P19F
        -- Decoder Input 0:1
        link_data_i(3) <= link_data_in(2);  -- Input Link 2
        link_data_i(2) <= link_data_in(3);  -- Input Link 3
        -- Decoder Input 0:0
        link_data_i(1) <= link_data_in(0);  -- Input Link 0
        link_data_i(0) <= link_data_in(1);  -- Input Link 1
      when "10" =>    -- 160MHz  Mapping from BOC Routing RX Section Document
        link_data_i(3) <= link_data_in(2);  -- Input Link 2 : DTO2 re
        link_data_i(2) <= link_data_in(0);  -- Input Link 0 : DTO  re
        link_data_i(1) <= link_data_in(3);  -- Input Link 3 : DTO2 fe
        link_data_i(0) <= link_data_in(1);  -- Input Link 1 : DTO  fe
--      when "10" =>    -- 160MHz  Mapping from BOC Routing RX Section Document Pre Upgrade
--        link_data_i(3) <= link_data_in(0);  -- Input Link 0 : DTO2 re
--        link_data_i(2) <= link_data_in(2);  -- Input Link 2 : DTO  re
--        link_data_i(1) <= link_data_in(1);  -- Input Link 1 : DTO2 fe
--        link_data_i(0) <= link_data_in(3);  -- Input Link 3 : DTO  fe
      when others =>  -- 40MHz & 80MHz
        link_data_i <= link_data_in; 
    end case;
  end if;
end process;

fifo_signal_map : process (
  rst_n_in,
  clk_in
  )
begin
  if (rst_n_in = '0') then
    for i in 0 to 3 loop
      fifo_data_i(i) <= (others => '0');
    end loop;
    fifo_wen_i             <= (others => '0'); 
    normal_not_raw_out     <= (others => '0'); 
    decoder_wr_trailer_out <= (others => '0');
  elsif (clk_in'event AND clk_in = '1') then
    case decode_type_in is
      when   "01" =>
        fifo_data_i(0)         <= fifo_data_d80_i(0);
        fifo_data_i(1)         <= fifo_data_d80_i(1);
        fifo_data_i(2)         <= (others => '0');
        fifo_data_i(3)         <= (others => '0');
        fifo_wen_i             <= "00" & dec80_fifo_wen_i;
        normal_not_raw_out     <= "00" & dec80_nnr_i;
        decoder_wr_trailer_out <= "00" & dec80_wr_trailer_i;
      when   "10" =>
        fifo_data_i(0)         <= fifo_data_d160_i;
        fifo_data_i(1)         <= (others => '0');
        fifo_data_i(2)         <= (others => '0');
        fifo_data_i(3)         <= (others => '0');
        fifo_wen_i             <= "000" & dec160_fifo_wen_i;
        normal_not_raw_out     <= "000" & dec160_nnr_i;
        decoder_wr_trailer_out <= "000" & dec160_wr_trailer_i;
      when others =>
        fifo_data_i            <= fifo_data_d40_i;
        fifo_wen_i             <= dec40_fifo_wen_i;
        normal_not_raw_out     <= dec40_nnr_i;
        decoder_wr_trailer_out <= dec40_wr_trailer_i;
    end case;
  end if;
end process;


header_trailer_limit_comparator : process (
  rst_n_in,
  clk_in
  )
begin
  if (rst_n_in = '0') then
    ht_limit_i <= (others => '0');
  elsif (clk_in'event AND clk_in = '1') then
    for i in 0 to 3 loop
      if (occ_count_i(i) >= ht_limit_in(10 downto 0)) then
        ht_limit_i(i) <= '1';
      else
        ht_limit_i(i) <= '0';
      end if;      
    end loop;
  end if;
end process header_trailer_limit_comparator;
     
rod_busy_limit_comparator : process (
  rst_n_in,
  clk_in
  )
begin
  if (rst_n_in = '0') then
    rb_limit_i <= (others => '0');
  elsif (clk_in'event AND clk_in = '1') then
    for i in 0 to 3 loop
      if (occ_count_i(i) >= rod_busy_limit_in(10 downto 0)) then
        rb_limit_i(i) <= '1';
      else
        rb_limit_i(i) <= '0';
      end if;      
    end loop;
  end if;
end process rod_busy_limit_comparator;
     
end rtl;

