--------------------------------------------------------------------------
--
-- Particle Physics Detector Electronics Support
-- University of Wisconsin
-- Lawrence Berkeley National Laboratory (c) 2000
-- ReadOutDriver Electronics
--
--------------------------------------------------------------------------
--
-- Filename: 256x16fifo.vhd                  
-- Description:
--  Input registered and Output registered fifo.     
--
--  The following state machines are all 1 stage glitch free MOORE MACHINES.
--   as per the LBL template  "vhdl/sm1_sample.vhd" code.
--
--                                                                   
--            +----------+   +---------------+      +-----------+    
-- data_in ---| Input    |---| Dual Port     |------| Output    |--- data_out
--            | Register |   | SRAM          |      | State     |     
--            |          |   |               |      | Logic     |     
--            |          |   |               |      |           |     
--            |          |   |               |      |           |     
--       +---->   en     |--->               |------>   en      |     
--       |    +----------+   |               |      +-----------+     
-- clk --+        |          +---------------+          |
-- wren ----------+            |                        | 
--                |            |      +-----------+     |                   
--                |            +------| Control   |-----+
--                |                   | Logic     |                        
--                +-------------------|           |                       
-- rden ------------------------------|           |                        
--                                    |           |                      
--                                    +-----------+                       
--                                                      
--
--------------------------------------------------------------------------
-- Structure: 
--  Multiple state machines in a flat topology/hierarchy which communicate
--   with single wire signals.  
--------------------------------------------------------------------------
-- Timing:
--    *The Engine FPGA runs at 40MHz => clk
--    *The timing structure of all engine code will be:
--      1.  Perform all logic operations on positive edge of clk40
--      2.  Perform all memory operations on positive edge of clk40
--------------------------------------------------------------------------

-- Author: Mark L. Nagel
-- Board Engineer: John M. Joseph
-- History:
--    23 February 2000 - MLN first version : requirements/functionality finally
--                       understood and sufficiently documented to proceed 
--                       with coding.
--
--------------------------------------------------------------------------

--------------------------------------------------------------------------
-- LIBRARY INCLUDES
--------------------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;  -- needed for logic operations
use IEEE.std_logic_arith.all; -- needed for +/- operations
--use IEEE.std_logic_unsigned.all;
--------------------------------------------------------------------------
-- PORT DECLARATION
--------------------------------------------------------------------------
entity fifo_256x16 is  -- 256x16 FIFO
  port (
     clk_in          : in    std_logic; -- clk40 input
     rst_n_in        : in    std_logic; -- asynchronous global reset
     wren_in         : in    std_logic; -- enable input register
     rden_in         : in    std_logic; -- enable output register
     data_in         : in    std_logic_vector(15 downto 0); --
     data_out        : out   std_logic_vector(15 downto 0); --
     full_flag_out   : out   std_logic; --
     empty_flag_out  : out   std_logic;
     word_count_out  : out   std_logic_vector(7 downto 0)
     );
end fifo_256x16;

architecture rtl of fifo_256x16 is
--------------------------------------------------------------------------
-- SIGNAL DECLARATION
--------------------------------------------------------------------------

signal rst_i             : std_logic;

signal ingoing_data_i    : std_logic_vector(15 downto 0);
signal outcoming_data_i  : std_logic_vector(15 downto 0);
signal last_wr_data_i    : std_logic_vector(15 downto 0);
signal last_rd_addr_i    : std_logic_vector( 7 downto 0);

signal wren_i            : std_logic;
signal write_en_i        : std_logic;
signal writein_adr_i     : unsigned(7 downto 0);

signal read_en_i         : std_logic;
signal readout_adr_i     : unsigned(7 downto 0);
signal occupancy_count_i : unsigned(7 downto 0);

signal mcc_skipped_count : unsigned(3 downto 0);
signal clr_mcc_skipped_count : std_logic;
signal last_event_id : unsigned(11 downto 0);
signal last_waddr : unsigned( 7 downto 0);

--------------------------------------------------------------------------
-- COMPONENT DECLARATION
--------------------------------------------------------------------------

component dpram_256x16  -- 256x16 DPRAM Block
  port (
    clk              : in  std_logic;
    rst              : in  std_logic;
    dprm_writein_bus : in  std_logic_vector(15 downto 0);
    dprm_readiin_bus : out std_logic_vector(15 downto 0);
    writein_adr      : in  unsigned(7 downto 0);
    readout_adr      : in  unsigned(7 downto 0);
    write_strobe     : in  std_logic;
    read_strobe      : in  std_logic;
    dprm_readout_bus : out std_logic_vector(15 downto 0)
    );
end component;

--------------------------------------------------------------------------
-- COMPONENT INSTANTIATION
--------------------------------------------------------------------------
begin

MB0 : dpram_256x16  -- 256x16 DPRAM Block
  port map(
    clk              => clk_in,
    rst              => rst_i,
    dprm_writein_bus => ingoing_data_i,
    dprm_readiin_bus => open, 
    writein_adr      => writein_adr_i,
    readout_adr      => readout_adr_i,
    write_strobe     => write_en_i,
    read_strobe      => read_en_i,
    dprm_readout_bus => outcoming_data_i
    );

--------------------------------------------------------------------------
-- PROCESS DECLARATION
--------------------------------------------------------------------------

rst_i <= not rst_n_in;
word_count_out <= std_logic_vector(occupancy_count_i);

-- input  address counter
input_counter : process (
  rst_n_in, 
  clk_in,
  write_en_i
  )
begin
  if (rst_n_in = '0') then
    writein_adr_i  <= (others => '0');
  elsif (clk_in'event and clk_in = '1') then
    if (write_en_i = '1' AND occupancy_count_i < 15) then
--    if (write_en_i = '1') then
      writein_adr_i <= writein_adr_i + 1;
    elsif (clr_mcc_skipped_count = '1') then
      writein_adr_i <= writein_adr_i + 1;
    elsif (mcc_skipped_count /= 0 AND wren_i = '0') then
      writein_adr_i <= last_waddr;
    end if; 
  end if;
end process input_counter;

-- output  address counter
output_counter : process (
  rst_n_in, 
  clk_in
  )
begin
  if (rst_n_in = '0') then
    readout_adr_i  <= (others => '0');
  elsif (clk_in'event and clk_in = '1') then
--    if (rden_in = '1' AND writein_adr_i > readout_adr_i) then
    if (rden_in = '1') then
      readout_adr_i <= readout_adr_i + 1;
    end if;
  end if;
end process output_counter;

mcc_skipped_events : process (
  rst_n_in, 
  clk_in
  )
begin
  if (rst_n_in = '0') then
    mcc_skipped_count <= (others => '0');
    last_event_id <= (others => '0');
    clr_mcc_skipped_count <= '0';
  elsif (clk_in'event and clk_in = '1') then
    clr_mcc_skipped_count <= '0';
    if (clr_mcc_skipped_count = '1') then
      mcc_skipped_count <= (others => '0');
    elsif (occupancy_count_i >= 15 AND wren_i = '1') then
      mcc_skipped_count <= mcc_skipped_count + '1';
    end if;
    if (write_en_i = '1') then
      last_event_id <= unsigned(data_in(11 downto 0)) - '1';
      last_waddr <= writein_adr_i;
    end if;
    if (mcc_skipped_count /= 0 AND wren_i = '0' AND clr_mcc_skipped_count = '0') then
      clr_mcc_skipped_count <= '1';
    end if; 
  end if;
end process mcc_skipped_events;

-- fifo 16bit word occupancy counter
occupancy_counter : process (
  rst_n_in, 
  clk_in,
  occupancy_count_i
  )
begin
  if (rst_n_in = '0') then
    occupancy_count_i  <= (others => '0');
    empty_flag_out <= '1';
    full_flag_out  <= '0'; 
  elsif (clk_in'event and clk_in = '1') then
--  else
    if (occupancy_count_i > 14) then 
      full_flag_out <= '1';
    else
      full_flag_out <= '0'; 
    end if;

    if (occupancy_count_i = 0) then 
      empty_flag_out <= '1';
    else
      empty_flag_out <= '0';
    end if;
    occupancy_count_i <= writein_adr_i - readout_adr_i;
  end if;
end process occupancy_counter;

-- delay wren to ram block by 1 clk
-- delay rden to ram block by 1 clk
-- read/write enable delay register
enable_delay : process (
  rst_n_in,
  clk_in,
  rden_in,
  wren_in,
  occupancy_count_i
  )
begin
  if (rst_n_in = '0') then
    read_en_i  <= '0';
    write_en_i <= '0';
    last_wr_data_i <= (others => '0');
  elsif (clk_in'event and clk_in = '1') then
    read_en_i  <= rden_in;--
    wren_i <= wren_in;
    if (mcc_skipped_count /= 0 AND wren_i = '0' AND write_en_i = '0') then
      write_en_i <= '1';
      last_wr_data_i <= std_logic_vector(mcc_skipped_count & last_event_id);
    elsif (writein_adr_i - readout_adr_i < 15) then 
      write_en_i <= wren_in;--
    else
      write_en_i <= '0';
    end if;
  end if;
end process enable_delay;

-- input  register
signal_register : process (
  rst_n_in, 
  clk_in, 
  ingoing_data_i, 
  data_in, 
  outcoming_data_i
  )
begin
  if (rst_n_in = '0') then
    data_out <= (others => '0');
    ingoing_data_i <= (others => '0');
elsif (clk_in'event and clk_in = '1') then
    data_out <= outcoming_data_i;
    if (mcc_skipped_count /= 0 AND wren_i = '0') then
      ingoing_data_i <= std_logic_vector(mcc_skipped_count & last_event_id);
    else
      ingoing_data_i <= data_in;
    end if; 
  end if;
end process signal_register;

end rtl; -- end code for tfsxstfifo