-------------------------------------------------------------------------------
-- Particle Physics Detector Electronics Support 
-- University of Wisconsin
-- Lawrence Berkeley National Laboratory (c) 1999
-- ReadOutDriver Electronics
--
-------------------------------------------------------------------------------
-- Filename: mb_data_decoder.vhd
-- Description:
--
-------------------------------------------------------------------------------
-- Structure: 
-------------------------------------------------------------------------------
-- Timing:
--    *The Engine FPGA runs at 40MHz => clk40
--    *The timing structure of all engine code will be:
--      1.  Perform all logic operations on positive edge of clk40
--      2.  Perform all memory operations on positive edge of clk40
-------------------------------------------------------------------------------
-- Author:         John Joseph
-- Board Engineer: 
-- History:
--  v13e : 01/28/2004 attached mb_fifo_ren_in directly to the FIFO block
--
-------------------------------------------------------------------------------

-------------------------------------------------------------------------------
-- LIBRARY INCLUDES
-------------------------------------------------------------------------------

library IEEE;
use IEEE.std_logic_1164.all;  -- needed for logic operations
use IEEE.std_logic_arith.all; -- needed for +/- operations

-------------------------------------------------------------------------------
-- PORT DECLARATION
-------------------------------------------------------------------------------
entity mb_data_decode is
  port(
    clk_in           : in  std_logic; -- clk40 input
    rst_n_in         : in  std_logic; -- asynchronous global reset
    mode_bits_in     : in  std_logic_vector(11 downto 0);
    mb_fifo_wen_n_in : in  std_logic; -- mb fifos wr strobe: active low
    mb_fifo_ren_in   : in  std_logic; -- mb fifos rd strobe: active high
    mb_fifo_rst_n_in : in  std_logic; -- modebit fifos reset strobe
    mode_bits_out    : out std_logic_vector(23 downto 0);
    mb_ready_out     : out std_logic; 
    mb_fifo_ef_n_out : out std_logic; -- readout mode fifo is empty
    mb_fifo_ff_n_out : out std_logic  -- readout mode fifo is full 
    );
end mb_data_decode; 

architecture rtl of mb_data_decode is

-------------------------------------------------------------------------------
-- SIGNAL DECLARATION
-------------------------------------------------------------------------------

signal mode_bits_i   : std_logic_vector(11 downto 0);
signal link_mb_i     : std_logic_vector(23 downto 0);
signal mb_fifo_wen_i : std_logic;
signal mb_data_set_i : std_logic;
signal mb_fifo_rst_i : std_logic;

signal wr_mb_to_fifo_i : std_logic;
signal mb_fifo_wen_dly : std_logic;

signal fifo_ef_i : std_logic;
signal fifo_ff_i : std_logic;

--------------------------------------------------------------------------
-- COMPONENT DECLARATION
--------------------------------------------------------------------------

component fifo_512x24
  port(
    clk_in              : in  std_logic; -- clk40 input
    rst_n_in            : in  std_logic; -- negative logic asynchronous global reset
    flush_fifo_in       : in  std_logic; -- flush fifo
    wren_in             : in  std_logic; -- enable input register
    rden_in             : in  std_logic; -- enable output register
    fifo_in             : in  std_logic_vector(23 downto 0); --
    fifo_out            : out std_logic_vector(23 downto 0); --
    full_flag_out       : out std_logic; --
    empty_flag_out      : out std_logic; --
    occupancy_count_out : out unsigned(8 downto 0)   --
   );
end component;

--------------------------------------------------------------------------
-- COMPONENT INSTANTIATION
--------------------------------------------------------------------------
begin

U1 : fifo_512x24
  port map(
    clk_in              => clk_in,
    rst_n_in            => rst_n_in,
    flush_fifo_in       => mb_fifo_rst_i,
    wren_in             => mb_fifo_wen_dly,
    rden_in             => mb_fifo_ren_in,
    fifo_in             => link_mb_i,
    fifo_out            => mode_bits_out,
    full_flag_out       => fifo_ff_i,
    empty_flag_out      => fifo_ef_i,
    occupancy_count_out => open
    );

--------------------------------------------------------------------------
-- PROCESS DECLARATION
--------------------------------------------------------------------------
signal_register : process (
  rst_n_in,
  clk_in
  )
begin
  if (rst_n_in = '0') then
    mode_bits_i   <= (others => '0');
    mb_fifo_wen_i    <= '0';
    mb_fifo_rst_i    <= '1';
    mb_fifo_ef_n_out <= '0';
    mb_fifo_ff_n_out <= '1';
  elsif (clk_in'event AND clk_in = '1') then
    mode_bits_i      <=     mode_bits_in;
    mb_fifo_wen_i    <= NOT mb_fifo_wen_n_in;
    mb_fifo_rst_i    <= NOT mb_fifo_rst_n_in;
    mb_fifo_ef_n_out <= NOT fifo_ef_i;
    mb_fifo_ff_n_out <= NOT fifo_ff_i;
  end if;
end process;    

mb_decode : process (
  rst_n_in,
  clk_in,
  mode_bits_i, 
  mb_fifo_wen_i, 
  mb_fifo_rst_i
  )
begin
  if (rst_n_in = '0') then
    link_mb_i(23 downto 0) <= (others => '0');
    mb_data_set_i   <= '0';
    wr_mb_to_fifo_i <= '0';
  elsif (clk_in'event AND clk_in = '1') then
    wr_mb_to_fifo_i <= '0';
    if (mb_fifo_rst_i = '1') then
      link_mb_i(23 downto 0) <= (others => '0');
      mb_data_set_i   <= '0';
    elsif (mb_fifo_wen_i = '1') then
      if (mb_data_set_i = '0') then
        for j in 0 to 11 loop
          link_mb_i(2*j) <= mode_bits_i(j);
        end loop;
        mb_data_set_i <= NOT mb_data_set_i;
      elsif (mb_data_set_i = '1') then
        for j in 0 to 11 loop
          link_mb_i((2*j)+1) <= mode_bits_i(j);
        end loop;
        mb_data_set_i <= NOT mb_data_set_i;
        wr_mb_to_fifo_i <= '1';
      end if;
    end if;
  end if;    
end process mb_decode;

mb_strobe : process (
  rst_n_in,
  clk_in,
  wr_mb_to_fifo_i
  )
begin
  if (rst_n_in = '0') then
    mb_fifo_wen_dly <= '0';
    mb_ready_out    <= '0';
  elsif (clk_in'event AND clk_in = '1') then
    mb_fifo_wen_dly <= wr_mb_to_fifo_i;
    mb_ready_out    <= mb_fifo_wen_dly;
  end if;
end process mb_strobe;

end rtl;

