-------------------------------------------------------------------------------
-- Lawrence Berkeley National Labs
-- ATLAS ROD ELECTRONICS
-------------------------------------------------------------------------------
-- Filename:
--    link_formatter_d80.vhd
-- Description:
--    ROD Front End Decoder
--       Four   Channel
--       PIXEL  Version
--
-- This file contains the serial decoding state machine for the PIXEL front
-- end. You should refer to three documents to better understand how the
-- vhdl for the decoder works.
--
--    1) "MCC-D2 Specifications- Output Data Format" presents a state 
--    transition diagram that shows how to decode the MCC serial data. 
--    The VHDL logic for the PIXEL decoder is closely based on this diagram.
--
-- Note: There should also be numerous other documents describing the PIXEL
-- decoder and its output data stream format.
--
-- In general, the PIXEL decoder must be robust to errors (i.e. cannot
-- go into an undefined state regardless of input data). Therefore,
-- error handling is also important and this adds to the
-- complexity/amount of logic in the decoder.

-------------------------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
-------------------------------------------------------------------------------
entity link_formatter_d80 is
  port (
    clk_in                     : in  std_logic;
    rst_n_in                   : in  std_logic;
    almost_full_in             : in  std_logic;
    half_clk_err_mask_in       : in  std_logic;
    enable_in                  : in  std_logic;
    config_mode_in             : in  std_logic;
    link_in                    : in  std_logic_vector(1 downto 0);
    normal_not_raw_out         : out std_logic;
    data_strobe_out            : out std_logic;
    decoder_writes_trailer_out : out std_logic;
    decoded_data_out           : out std_logic_vector(31 downto 0);
    decoder_state_in_idle      : out std_logic
    );
end link_formatter_d80;
-------------------------------------------------------------------------------

architecture rtl of link_formatter_d80 is

-------------------------------------------------------------------------------
-- SIGNAL DECLARATIONS
-------------------------------------------------------------------------------
  type pixel_state_typedef is (
    idle,
    store_id,
    write_id,
    mcc_fe_test,
    fe_num,
    test,
    hit,
    mcc_flag,
    fe_flag,
    raw_data_header,
    first_raw_data,
    trap_raw_data,
    wr_raw_data,
    wr_last_word,
    wait_for_trailer,
    trailer,
    send_trailer
    );
  signal dpar_state : pixel_state_typedef;

  signal almost_full_i       : std_logic;
  signal dpar_hit            : std_logic_vector(31 downto 0);
  signal start_write         : std_logic;
  signal wr_header           : std_logic;
  signal wr_trailer          : std_logic;
  signal serial_data_reg     : std_logic_vector(31 downto 0);
  signal link_sh             : std_logic_vector( 3 downto 0);
  signal fe_reg              : std_logic_vector( 3 downto 0);
  signal header_sh_reg       : std_logic_vector( 9 downto 0);
  signal header_detect_i     : std_logic;
  signal header_error_i      : std_logic;
  signal header_error_flag_i : std_logic;
  signal link_in_i           : std_logic_vector( 1 downto 0);
  signal trailer_detect_i    : std_logic;
  signal trailer_error_i     : std_logic;

begin
-------------------------------------------------------------------------------
-- Process Declaration
-------------------------------------------------------------------------------
input_register : process (
  clk_in, 
  rst_n_in
  )
begin
  if (rst_n_in = '0') then
    almost_full_i <= '0' ;
    link_in_i     <= (others => '0');
-- pragma translate_off
    serial_data_reg <= (others => '0');
-- pragma translate_on
  elsif (clk_in'event AND clk_in = '1') then
    almost_full_i <= almost_full_in;
    if (enable_in = '1') then
      link_in_i <= link_in;
    else
      link_in_i <= (others => '0');
    end if;  
    serial_data_reg <= serial_data_reg(29 downto 0) & link_in_i;
  end if;
end process input_register;

-------------------------------------------------------------------------------
-- Process to detect Link Headers
find_header : process (
  clk_in, 
  rst_n_in
  )
begin  -- find header
  if (rst_n_in = '0') then
-- pragma translate_off
    header_sh_reg  <= (others => '0');
-- pragma translate_on
    header_error_i <= '0';
  elsif (clk_in'event AND clk_in = '1') then
-- Correct header pattern is 11101.  A header can also have up to one bit 
-- of error in any location.         
    header_sh_reg <= header_sh_reg(7 downto 0) & serial_data_reg(13 downto 12);
    if (dpar_state = idle) then
      case header_sh_reg(8 downto 3) is
        when "011101" =>      -- "111101" is a possible hit pattern with an error 
          header_detect_i <= '1';
          header_error_i  <= '0';

        when "111101" =>      -- "111101" is a possible hit pattern with an error 
          header_detect_i <= '1';
          header_error_i  <= NOT half_clk_err_mask_in; -- mask in half clock mode

        when "001101" =>
          header_detect_i <= '1';
          header_error_i  <= '1';

        when "010101" =>
          header_detect_i <= '1';
          header_error_i  <= '1';

        when "011001" =>
          header_detect_i <= '1';
          header_error_i  <= '1';

        when "011111" =>
          header_detect_i <= NOT header_sh_reg(2);
          header_error_i  <= NOT header_sh_reg(2);

        when "011100" =>
          header_detect_i <= '1';
          header_error_i  <= '1';

        when others =>
          header_detect_i <= '0';
          header_error_i  <= '0';
      end case;
    else
      header_detect_i <= '0';
      header_error_i  <= '0';
    end if;
  end if;
end process find_header;
  
-------------------------------------------------------------------------------
-- Process to detect trailers: Only used if Formatter goes into Raw Data Mode
find_trailer : process (
  clk_in, 
  rst_n_in
  )    
begin -- find_trailer
  if (rst_n_in = '0') then
    trailer_detect_i <= '0';
    trailer_error_i  <= '0';
  elsif (clk_in'event AND clk_in = '1') then
    if (serial_data_reg(21 downto 0) = "0000000000000000000000") then
      trailer_detect_i <= '1';
      trailer_error_i  <= '0';
--    elsif (serial_data_reg(21 downto 0) = "1111111111111111111111") then
--      trailer_detect_i <= '1';
--      trailer_error_i  <= NOT config_mode_in;
    else
      trailer_detect_i <= '0';
      trailer_error_i  <= '0';
    end if;
  end if;  
end process find_trailer;

-------------------------------------------------------------------------------
-- Serial to parallel decoder <==> 80MHz Input to ROD
pixel80_data_parser : process (
  rst_n_in,
  clk_in
  )
variable wait_count    : unsigned(3 downto 0);
variable sync_count_to : unsigned(3 downto 0);
variable sync_bit_loc  : std_logic;
begin
  if (rst_n_in = '0') then
    dpar_state	  <= idle;
    dpar_hit     <= (others => '0');
    wait_count    := (others => '0');
    sync_count_to := (others => '0');
    sync_bit_loc  := '0';
    start_write   <= '0';
    wr_header     <= '0';
    wr_trailer    <= '0';
    header_error_flag_i <= '0';
    decoder_state_in_idle <= '1';
  elsif (clk_in'EVENT AND clk_in = '1') then
    start_write <= '0';
    wr_header   <= '0';
    wr_trailer  <= '0';
    case dpar_state is
      when idle =>
        sync_count_to := (others => '0');
        wait_count    := (others => '0');
        sync_bit_loc  := '0';
        if (enable_in = '1') then
          if (header_detect_i = '1' OR  -- wake up on mci header detect
              (config_mode_in = '1' AND header_sh_reg(9 downto 8) /= "00")) then -- wake up on first rising edge aligned to a header
            header_error_flag_i <= header_error_i;
            dpar_state <= store_id;
          end if;
        end if;
        decoder_state_in_idle <= '1';

      when store_id =>  -- store L1ID
        dpar_hit(31 downto 29) <= "001";
        dpar_hit(27 downto 16) <= (others => '0');
        if (serial_data_reg(12) = '0' OR config_mode_in = '1') then
          if (config_mode_in = '0') then
            dpar_hit(15 downto 8) <= (others => '0');
          else
            dpar_hit(15 downto 8) <= serial_data_reg(26 downto 19);
          end if;
          if (config_mode_in = '1') then
            dpar_hit(28)         <= '0';
            dpar_hit(7 downto 0) <= X"CD";
          else
            dpar_hit(28)         <= header_error_flag_i;
            dpar_hit(7 downto 0) <= X"ED";
          end if;
          dpar_state <= raw_data_header;
        elsif (serial_data_reg(12) = '1') then
          dpar_hit(15 downto  8) <= serial_data_reg(20 downto 13);
          dpar_hit(7 downto 0) <= serial_data_reg(11 downto 4);
          dpar_state <= write_id;
        end if;
        decoder_state_in_idle <= '0';
 
      when write_id =>     -- sync bit loc 5
        sync_bit_loc  := '0';
        sync_count_to := "0011";
        wr_header     <= '1';
        dpar_state    <= mcc_fe_test;

      when mcc_fe_test =>  -- sync bit loc 7 (In this state once per event)
        if (wait_count = sync_count_to) then
--  ******* Decode section from V20 to V2F ****************************
--          if (serial_data_reg(13 downto 9) = "11110" AND serial_data_reg(4) = '1') then
--            fe_reg     <= serial_data_reg(8 downto 5);
--            dpar_state <= test;
--          else
--            dpar_hit(31 downto 28) <= X"6";
--            dpar_hit(27 downto  0) <= serial_data_reg(27 downto 0);
--            wait_count  := (others => '0');
--            dpar_state <= wr_raw_data;
--          end if;
--  *******************************************************************
          if (serial_data_reg(13 downto 9) = "11110") then
            if (serial_data_reg(4) = '1') then  -- Sync Bit detected
              fe_reg     <= serial_data_reg(8 downto 5);
              dpar_state <= test;
            else  -- NO Sync Bit
              dpar_hit(31 downto 28) <= X"6";
              dpar_hit(27 downto  0) <= serial_data_reg(27 downto 0);
              wait_count  := (others => '0');
              dpar_state <= wr_raw_data;
            end if;
          else
            dpar_state <= wait_for_trailer;
          end if;
          sync_count_to := (others => '0');
          sync_bit_loc  := '0';
        else
          wait_count := wait_count + 1;
        end if;

      when fe_num =>
        if (sync_bit_loc = '0') then
          if (wait_count = sync_count_to) then
            if (serial_data_reg(4) = '1') then
              fe_reg     <= serial_data_reg(8 downto 5);
              dpar_state <= test;
            else 
              dpar_hit(31 downto 28) <= X"6";
              dpar_hit(27 downto  0) <= serial_data_reg(27 downto 0);
              wait_count  := (others => '0');
              dpar_state <= wr_raw_data;
            end if;
          else
            wait_count := wait_count + 1;
          end if;
        elsif (sync_bit_loc = '1') then
          if (wait_count = sync_count_to) then
            if (serial_data_reg(5) = '1') then
              fe_reg     <= serial_data_reg(9 downto 6);
              dpar_state <= test;
            else 
              dpar_hit(31 downto 28) <= X"6";
              dpar_hit(27 downto  0) <= serial_data_reg(27 downto 0);
              wait_count  := (others => '0');
              dpar_state <= wr_raw_data;
            end if;
          else
            wait_count := wait_count + 1;
          end if;
        end if;

      when test =>  -- test for next data type
        wait_count := (others => '0');
--  Test here for:  FE_FLAG  => "s11110ffffmmmm"
--                  MCC_FLAG => "s11111ffffffffmmmmmmmm"
--                  FE_NUM   => "s1110nnnn"
        if (sync_bit_loc = '0') then
          if (serial_data_reg(6 downto 1) = "111110") then
            dpar_state <= fe_flag;
            sync_count_to := "0101";
            sync_bit_loc  := '0';
          elsif (serial_data_reg(6 downto 1) = "111111") then
            dpar_state <= mcc_flag;
            sync_count_to := "1001";
            sync_bit_loc  := '0';
          elsif (serial_data_reg(6 downto 2) = "11110") then
            dpar_state <= fe_num;
            sync_count_to := "0011";
            sync_bit_loc  := '1';
          else
            dpar_state <= hit; 
            sync_count_to := "1001";
            sync_bit_loc  := '0';
          end if;
        elsif (sync_bit_loc = '1') then
          if (serial_data_reg(7 downto 2) = "111110") then
            dpar_state <= fe_flag;
            sync_count_to := "0101";
            sync_bit_loc  := '1';
          elsif (serial_data_reg(7 downto 2) = "111111") then
            dpar_state <= mcc_flag;
            sync_count_to := "1001";
            sync_bit_loc  := '1';
          elsif (serial_data_reg(7 downto 3) = "11110") then
            dpar_state <= fe_num;
            sync_count_to := "0010";
            sync_bit_loc  := '0';
          else
            dpar_state <= hit; 
            sync_count_to := "1001";
            sync_bit_loc  := '1';
          end if;
        end if;

      when hit =>
        if (sync_bit_loc = '0') then
          if (wait_count = sync_count_to) then
            if (serial_data_reg(4) = '1') then
              dpar_hit(31 downto 28) <= "1000";
              dpar_hit(27 downto 24) <= fe_reg(3 downto 0);
              dpar_hit(23 downto 16) <= serial_data_reg(12 downto  5);  -- TOT
              dpar_hit(15 downto 13) <= (others => '0');
              dpar_hit(12 downto  8) <= serial_data_reg(17 downto 13);  -- COL
              dpar_hit( 7 downto  0) <= serial_data_reg(25 downto 18);  -- ROW
              dpar_state  <= test;
              start_write <= '1';
            elsif (serial_data_reg(25 downto 4) = "0000000000000000000000") then -- Trailer
              dpar_hit(31 downto 28) <= "0100";
              dpar_hit(27)           <= almost_full_i;
              dpar_hit(26 downto  0) <= (others => '0');
              wr_trailer             <= '1';
              dpar_state             <= idle;
            else 
              dpar_hit(31 downto 28) <= X"6";
              dpar_hit(27 downto  0) <= serial_data_reg(27 downto 0);
              wait_count  := (others => '0');
              dpar_state <= wr_raw_data;
            end if;
          else
            wait_count := wait_count + 1;
          end if;
        elsif (sync_bit_loc = '1') then
          if (wait_count = sync_count_to) then
            if (serial_data_reg(5) = '1') then
              dpar_hit(31 downto 28) <= "1000";
              dpar_hit(27 downto 24) <= fe_reg(3 downto 0);
              dpar_hit(23 downto 16) <= serial_data_reg(13 downto  6);  -- TOT
              dpar_hit(15 downto 13) <= (others => '0');
              dpar_hit(12 downto  8) <= serial_data_reg(18 downto 14);  -- COL
              dpar_hit( 7 downto  0) <= serial_data_reg(26 downto 19);  -- ROW
              dpar_state  <= test;
              start_write <= '1';
            elsif (serial_data_reg(26 downto 5) = "0000000000000000000000") then -- Trailer
              dpar_hit(31 downto 28) <= "0100";
              dpar_hit(27)           <= almost_full_i;
              dpar_hit(26 downto  0) <= (others => '0');
              wr_trailer             <= '1';
              dpar_state             <= idle;
            else 
              dpar_hit(31 downto 28) <= X"6";
              dpar_hit(27 downto  0) <= serial_data_reg(27 downto 0);
              wait_count  := (others => '0');
              dpar_state <= wr_raw_data;
            end if;
          else
            wait_count := wait_count + 1;
          end if;
        end if;

      when fe_flag =>
        if (sync_bit_loc = '0') then
          if (wait_count = sync_count_to) then
            if (serial_data_reg(4) = '1') then
              dpar_hit(31 downto 28) <= "0000";
              dpar_hit(27 downto 24) <= fe_reg(3 downto 0);
              dpar_hit(23 downto 13) <= (others => '0');
              dpar_hit(12 downto  0) <= serial_data_reg(17 downto 5);
              dpar_state             <= test;
              start_write            <= '1';
            else 
              dpar_hit(31 downto 28) <= X"6";
              dpar_hit(27 downto  0) <= serial_data_reg(27 downto 0);
              wait_count  := (others => '0');
              dpar_state <= wr_raw_data;
            end if;
          else
            wait_count := wait_count + 1;
          end if;
        elsif (sync_bit_loc = '1') then
          if (wait_count = sync_count_to) then
            if (serial_data_reg(5) = '1') then
              dpar_hit(31 downto 28) <= "0000";
              dpar_hit(27 downto 24) <= fe_reg(3 downto 0);
              dpar_hit(23 downto 13) <= (others => '0');
              dpar_hit(12 downto  0) <= serial_data_reg(18 downto 6);
              dpar_state             <= test;
              start_write            <= '1';
            else 
              dpar_hit(31 downto 28) <= X"6";
              dpar_hit(27 downto  0) <= serial_data_reg(27 downto 0);
              wait_count  := (others => '0');
              dpar_state <= wr_raw_data;
            end if;
          else
            wait_count := wait_count + 1;
          end if;
        end if;
 
      when mcc_flag =>
        if (sync_bit_loc = '0') then
          if (wait_count = sync_count_to) then
            if (serial_data_reg(4) = '1') then
              dpar_hit(31 downto 28) <= "0001";
              dpar_hit(27 downto 24) <= fe_reg(3 downto 0);
              dpar_hit(23 downto 21) <= (others => '0');
              dpar_hit(20 downto  0) <= serial_data_reg(25 downto 5);
              dpar_state  <= test;
              start_write <= '1';
            else 
              dpar_hit(31 downto 28) <= X"6";
              dpar_hit(27 downto  0) <= serial_data_reg(27 downto 0);
              wait_count  := (others => '0');
              dpar_state <= wr_raw_data;
            end if;
          else
            wait_count := wait_count + 1;
          end if;
        elsif (sync_bit_loc = '1') then
          if (wait_count = sync_count_to) then
            if (serial_data_reg(5) = '1') then
              dpar_hit(31 downto 28) <= "0001";
              dpar_hit(27 downto 24) <= fe_reg(3 downto 0);
              dpar_hit(23 downto 21) <= (others => '0');
              dpar_hit(20 downto  0) <= serial_data_reg(26 downto 6);
              dpar_state  <= test;
              start_write <= '1';
            else 
              dpar_hit(31 downto 28) <= X"6";
              dpar_hit(27 downto  0) <= serial_data_reg(27 downto 0);
              wait_count  := (others => '0');
              dpar_state <= wr_raw_data;
            end if;
          else
            wait_count := wait_count + 1;
          end if;
        end if;

      when raw_data_header => 
        wr_header  <= '1';
        dpar_state <= first_raw_data;

      when first_raw_data =>
        if (wait_count = 1) then
          dpar_hit(31 downto 28) <= X"6"; 
          dpar_hit(27 downto 0) <= serial_data_reg(31 downto 4);
          dpar_state <= wr_raw_data;
          wait_count := (others => '0');
        end if;
        wait_count := wait_count + 1;

      when trap_raw_data =>
        if (wait_count = 13) then
          dpar_hit(31 downto 28) <= X"6"; 
          if (config_mode_in = '0') then
            dpar_hit(27 downto 0) <= serial_data_reg(27 downto 0);
          else
            dpar_hit(27 downto 0) <= serial_data_reg(31 downto 4);
          end if;
          dpar_state <= wr_raw_data;
          wait_count := (others => '0');
        elsif (trailer_detect_i = '1') then
          case wait_count is 
            when X"C" => dpar_hit(27 downto 0) <= serial_data_reg(29 downto 2);
                         dpar_state <= wr_last_word;
            when X"B" => dpar_hit(27 downto 0) <= serial_data_reg(27 downto 0);
                         dpar_state <= wr_last_word;
            when X"A" => dpar_hit(27 downto 0) <= serial_data_reg(25 downto 0) & "00";
                         dpar_state <= wr_last_word;
            when others => dpar_state <= trailer;
          end case;
          
        end if;
        wait_count := wait_count + 1;

      when wr_last_word =>
        start_write <= '1';
        dpar_state <= trailer;

      when wr_raw_data =>
        start_write <= '1';
        if (trailer_detect_i = '1') then
          dpar_state <= trailer;
        elsif (config_mode_in = '1') then
          dpar_state <= trap_raw_data;
        else
          dpar_state <= wait_for_trailer;
        end if;

      when wait_for_trailer =>
        if (trailer_detect_i = '1') then
          dpar_state <= trailer;
        end if;

      when trailer =>
        dpar_hit(31 downto 29) <= "010";
        dpar_hit(28)           <= trailer_error_i;
        dpar_hit(27)           <= almost_full_i;
        dpar_hit(26 downto  0) <= (others => '0');
        dpar_state             <= send_trailer;

      when send_trailer =>
        start_write <= '1';
        dpar_state  <= idle;

      when others =>
        dpar_state <= idle;
    end case;
  end if;
end process pixel80_data_parser;

-------------------------------------------------------------------------------
-- purpose: This process looks at the current state and outputs the data
-- words depending on which state and which bit is being shifted in
wr_output_data : process (
  clk_in, 
  rst_n_in
  )
begin  -- process
  if (rst_n_in = '0') then
    decoded_data_out   <= (others => '0');
    data_strobe_out    <= '0';
    normal_not_raw_out <= '1';
    decoder_writes_trailer_out <= '0';
  elsif (clk_in'event AND clk_in = '1') then
    decoded_data_out <= dpar_hit;
    data_strobe_out  <= (start_write AND NOT almost_full_i) OR 
                         wr_header OR wr_trailer;
    decoder_writes_trailer_out <= wr_trailer;

    if (dpar_state = wr_raw_data    OR
        dpar_state = trap_raw_data  OR
        dpar_state = first_raw_data OR
        dpar_state = raw_data_header) then
      normal_not_raw_out <= '0';
    else
      normal_not_raw_out <= '1';
    end if;    

  end if;
end process;
--decoded_data_out   <= dpar_hit;

end rtl;

