-------------------------------------------------------------------------------
-- Lawrence Berkeley National Labs
-- ATLAS ROD ELECTRONICS
-------------------------------------------------------------------------------
-- Filename:
--    link_formatter_layer2.vhd
-- Description:
--    ROD Front End Decoder
--       Four   Channel
--       PIXEL  Version
--
-- This file contains the serial decoding state machine for the PIXEL front
-- end. You should refer to three documents to better understand how the
-- vhdl for the decoder works.
--
--    1) "MCC-D2 Specifications- Output Data Format" presents a state 
--    transition diagram that shows how to decode the MCC serial data. 
--    The VHDL logic for the PIXEL decoder is closely based on this diagram.
--
-- Note: There should also be numerous other documents describing the PIXEL
-- decoder and its output data stream format.
--
-- In general, the PIXEL decoder must be robust to errors (i.e. cannot
-- go into an undefined state regardless of input data). Therefore,
-- error handling is also important and this adds to the
-- complexity/amount of logic in the decoder.

-------------------------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
-------------------------------------------------------------------------------
entity link_formatter_d40 is
  port (
    clk_in                     : in  std_logic;
    rst_n_in                   : in  std_logic;
    almost_full_in             : in  std_logic;  -- 1 => header/trailer mode
                                                 -- 0 => fifo ok
    enable_in                  : in  std_logic;  -- enable link formatter
    config_mode_in             : in  std_logic;
    half_clk_err_mask_in       : in  std_logic;
    link_in                    : in  std_logic;  -- front end serial data
    normal_not_raw_out         : out std_logic;  -- 1 => normal data mode,
                                                 -- 0 => raw data mode
    data_strobe_out            : out std_logic;  -- write new output word
    decoder_writes_trailer_out : out std_logic;  -- trailer being written
    decoded_data_out           : out std_logic_vector(31 downto 0);
    decoder_state_in_idle      : out std_logic
    );
end link_formatter_d40;
-------------------------------------------------------------------------------

architecture rtl of link_formatter_d40 is

-------------------------------------------------------------------------------
-- SIGNAL DECLARATIONS
-------------------------------------------------------------------------------
  type pixel_state_typedef is(
    idle,
    l1_id,
    bc_id_sync,
    bc_id,
    mcc_fe_test,
    test_all,
    trap_hit_data,
    hit,
    trap_mcc_flag,
    mcc_flag,
    trap_fe_flag,
    fe_flag,
    raw_data_header,
    first_raw_data,
    trap_raw_data,
    wr_raw_data,
    wait_for_trailer,
    trailer
    );
  signal pres_state : pixel_state_typedef;

  signal almost_full_flag_i       : std_logic;
  signal almost_full_i            : std_logic;
  signal bit_count_i              : unsigned (4 downto 0);
  signal data_out                 : std_logic_vector(31 downto 0);
  signal data_out_write_i         : std_logic;
  signal decoder_writes_trailer_i : std_logic;
  signal fe_reg                   : std_logic_vector(3 downto 0);
  signal header_detect_i          : std_logic;
  signal header_error_i           : std_logic;
  signal header_error_flag_i      : std_logic;
  signal header_sh_reg            : std_logic_vector(7 downto 0) ;
  signal link_in_i                : std_logic;
  signal normal_not_raw_i         : std_logic;
  signal reset_bit_count_i        : std_logic;
  signal serial_data_reg          : std_logic_vector(27 downto 0) ;
  signal trailer_counter_i        : unsigned (4 downto 0);
  signal trailer_detect_i         : std_logic;
  signal trailer_error_i          : std_logic;
  signal trailer_error_flag_i     : std_logic;
  
-------------------------------------------------------------------------------
-- Constant Declaration
-------------------------------------------------------------------------------

-------------------------------------------------------------------------------
-- Componenent Declaration
-------------------------------------------------------------------------------

begin
-------------------------------------------------------------------------------
-- Component Instantiation
-------------------------------------------------------------------------------

-------------------------------------------------------------------------------
-- ASSIGN SIGNALS
-------------------------------------------------------------------------------

decoder_writes_trailer_out <= decoder_writes_trailer_i;
normal_not_raw_out         <= normal_not_raw_i;
data_strobe_out            <= data_out_write_i;
decoded_data_out           <= data_out;

-------------------------------------------------------------------------------
-- Process Declaration
-------------------------------------------------------------------------------
input_register : process (
  clk_in, 
  rst_n_in
  )
begin
  if (rst_n_in = '0') then
    almost_full_i   <= '0' ;
    link_in_i       <= '0' ;
    serial_data_reg <= (others => '0');
  elsif (clk_in'event AND clk_in = '1') then
    almost_full_i   <= almost_full_in;
    link_in_i       <= link_in AND enable_in;
    serial_data_reg <= serial_data_reg(26 downto 0) & link_in_i;
  end if;
end process input_register;

-------------------------------------------------------------------------------
-- Process to detect Link Headers
find_header : process (
  clk_in, 
  rst_n_in
  )
begin  -- find header
  if (rst_n_in = '0') then
    header_sh_reg  <= (others => '0');
    header_error_i <= '0';
  elsif (clk_in'event AND clk_in = '1') then
-- Correct header pattern is 11101.  A header can also have up to one bit 
-- of error in any location. Look at serial_data_reg to determine data...
--  10101010101010101010
--  01010101010101010101
--  xxxxxxhhhhhllllbbbbs
--            A9876543210
    header_sh_reg <= header_sh_reg(6 downto 0) & serial_data_reg(10);
    case header_sh_reg is
      when "00011101" =>
        if (half_clk_err_mask_in = '1' AND serial_data_reg(1) = '0') then
          header_detect_i <= '0';
          header_error_i  <= '0';
        else
          header_detect_i <= '1';
          header_error_i  <= '0';
        end if;

      when "00111101" =>  -- "111101" is a possible hit pattern with an error 
        header_detect_i <= '1';
        header_error_i  <= NOT half_clk_err_mask_in; -- mask in half clock mode

      when "00001101" =>
        header_detect_i <= '1';
        header_error_i  <= '1';

      when "00010101" =>
        if (half_clk_err_mask_in = '1') then
          header_detect_i <= '0';
          header_error_i  <= '0';
        else
          header_detect_i <= '1';
          header_error_i  <= '1';
        end if;

      when "00011001" =>
        header_detect_i <= '1';
        header_error_i  <= '1';

      when "00011111" =>
        if (serial_data_reg(10 downto 0) = "11111111111") then
          header_detect_i <= '0';
          header_error_i  <= '0';
        else
          header_detect_i <= '1';
          header_error_i  <= '1';
        end if;

      when "00011100" =>
        header_detect_i <= '1';
        header_error_i  <= '1';

      when others =>
        header_detect_i <= '0';
        header_error_i  <= '0';
    end case;
  end if;
end process find_header;
  
-------------------------------------------------------------------------------
-- Process to detect Link Headers
find_trailer : process (
  clk_in, 
  rst_n_in
  )    
begin -- find_trailer
  if (rst_n_in = '0') then
    trailer_counter_i <= (others => '0');
    trailer_detect_i  <= '0';
    trailer_error_i   <= '0';
  elsif (clk_in'event and clk_in = '1') then
    -- If 1 shifts in and 0 shifts out, increment num 1's
    -- If 0 shifts in and 1 shifts out, decrement num 1's
    if (serial_data_reg(0) = '1' AND serial_data_reg(22) = '0') then
      trailer_counter_i <= trailer_counter_i + 1;
    elsif (serial_data_reg(0) = '0' AND serial_data_reg(22) = '1') then
      trailer_counter_i <= trailer_counter_i - 1;
    end if;

    if (std_logic_vector(trailer_counter_i) = "00000") then
      -- if 0 or 1 followed by 22 0's, a trailer is detected
      trailer_detect_i <= '1';
      -- Zero followed by 22 zeroes is a trailer w/ bit error
      trailer_error_i  <= NOT serial_data_reg(23) AND NOT config_mode_in;
      -- trailer with 1 or 0 followed by 22 zeroes
    else
      trailer_detect_i <= '0';
      trailer_error_i  <= '0';
    end if;
  end if;  
end process find_trailer;

-------------------------------------------------------------------------------
-- Serial to parallel decoder
pixel_data_decoder : process (
  rst_n_in,
  clk_in
  )
begin  -- process pxl_comb_fsm
  if (rst_n_in = '0') then
    pres_state <= idle;
    header_error_flag_i   <= '0';
    trailer_error_flag_i  <= '0';
    decoder_state_in_idle <= '1';
  elsif (clk_in'event and clk_in = '1') then
    reset_bit_count_i    <= '0';
    -- check all state transition conditions
    if (trailer_detect_i = '1'       AND 
        pres_state/= idle            AND 
        pres_state/= l1_id           AND 
        pres_state/= raw_data_header AND 
        pres_state/= trailer) then
      pres_state <= trailer;
      trailer_error_flag_i <= trailer_error_i;
    end if;
    
    case pres_state is
      when idle =>
        if (header_detect_i = '1' OR (config_mode_in = '1' AND header_sh_reg(5) = '1')) then
          header_error_flag_i <= header_error_i;
          pres_state <= l1_id;
        end if;
        fe_reg <= (others => '0');
        reset_bit_count_i <= '1';
        trailer_error_flag_i <= '0';
        decoder_state_in_idle <= '1';
        
      when l1_id =>
        if (config_mode_in = '1' OR serial_data_reg(4) = '0') then
          pres_state <= raw_data_header;
        elsif (serial_data_reg(4) = '1') then
          pres_state <= bc_id_sync;
          reset_bit_count_i <= '1';
        end if;
        decoder_state_in_idle <= '0';
                
      when bc_id_sync =>
        if (bit_count_i = 7) then
          if (serial_data_reg(4) = '1') then
            pres_state <= bc_id;
          else
            pres_state <= raw_data_header;
          end if;
        end if;

      when bc_id =>
        pres_state <= mcc_fe_test;
        reset_bit_count_i <= '1';

      when mcc_fe_test =>
        if (bit_count_i = 6) then
          if (serial_data_reg(13 downto 9) = "11110" AND serial_data_reg(4) = '1') then
            pres_state <= test_all;
            reset_bit_count_i <= '1';
            fe_reg <= serial_data_reg(8 downto 5);
          end if;
        end if;

      when test_all =>
        if (bit_count_i = 7) then
          if (serial_data_reg(14 downto 10) = "11110" AND serial_data_reg(5) = '1') then
            reset_bit_count_i <= '1';
            fe_reg <= serial_data_reg(9 downto 6);
          elsif (serial_data_reg(14 downto 9) = "111110" AND serial_data_reg(0) = '1') then
            pres_state <= trap_fe_flag;
          elsif (serial_data_reg(14 downto 9) = "111111") then
            pres_state <= trap_mcc_flag;
          else
            pres_state <= trap_hit_data;
          end if;
        end if;
      
      when trap_hit_data =>
        if (bit_count_i = 19 AND serial_data_reg(4) = '1') then
          pres_state <= hit;
        elsif (bit_count_i = 20 AND trailer_detect_i = '1' AND trailer_error_i = '0') then
          pres_state <= trailer;
          trailer_error_flag_i <= trailer_error_i; --
        elsif (bit_count_i = 21) then
          pres_state <= wr_raw_data;
        end if;

      when hit =>
        pres_state <= test_all;
        reset_bit_count_i <= '1';

      when trap_fe_flag =>
        if (bit_count_i = 11) then
          pres_state <= fe_flag;
        end if;

      when fe_flag =>
        pres_state <= test_all;
        reset_bit_count_i <= '1';

      when trap_mcc_flag =>
        if (bit_count_i = 19 AND serial_data_reg(4) = '1') then
          pres_state <= mcc_flag;
        elsif (bit_count_i = 21) then
          pres_state <= wr_raw_data;
        end if;

      when mcc_flag =>
        pres_state <= test_all;
        reset_bit_count_i <= '1';

      when trailer =>
        pres_state <= idle;

      when raw_data_header => 
        pres_state <= first_raw_data;
        reset_bit_count_i <= '1';

      when first_raw_data =>
        if (bit_count_i = 6) then
          pres_state <= wr_raw_data;
        end if;

      when trap_raw_data =>
        if (trailer_detect_i = '1' AND trailer_error_i = '0') then
          pres_state <= trailer;
          trailer_error_flag_i <= '0'; --
        elsif (bit_count_i = 25) then
          pres_state <= wr_raw_data;
        end if;

      when wr_raw_data =>
        if (config_mode_in = '1') then
          pres_state <= trap_raw_data;
          reset_bit_count_i <= '1';
        else
          pres_state <= wait_for_trailer;
          reset_bit_count_i <= '1';
        end if;

      when wait_for_trailer =>
        if (trailer_detect_i = '1' AND trailer_error_i = '0') then
          pres_state <= trailer;
          trailer_error_flag_i <= '0'; --
        end if;
        
      when others   => null;
    end case;
  end if;
end process pixel_data_decoder;

-------------------------------------------------------------------------------
-- Process to count data bits as the flow through the decoder
data_bit_counter : process (
  clk_in, 
  rst_n_in
  )
begin  -- process set_clocked_fsm
  if (rst_n_in = '0') then
    bit_count_i <= (others => '0');
  elsif (clk_in'event and clk_in = '1') then
    if (reset_bit_count_i = '1') then
      bit_count_i <= (others => '0');
    else
      bit_count_i <= bit_count_i + 1;
    end if;
  end if;
end process data_bit_counter;

-------------------------------------------------------------------------------
-- purpose: This process looks at the current state and outputs the data
-- words depending on which state and which bit is being shifted in
wr_output_data : process (
  clk_in, 
  rst_n_in
  )
begin  -- process
  if (rst_n_in = '0') then
    decoder_writes_trailer_i <= '0';
    data_out_write_i         <= '0';
    data_out                 <= (others => '0');
    normal_not_raw_i         <= '1';
  elsif (clk_in'event and clk_in = '1') then
    data_out_write_i <= '0';
    if (almost_full_i = '1') then
      almost_full_flag_i <= '1';
    end if;

    if (pres_state = wr_raw_data   OR
        pres_state = trap_raw_data OR
        pres_state = raw_data_header) then
      normal_not_raw_i <= '0';
    else
      normal_not_raw_i <= '1';
    end if;    

    case pres_state is
      when idle  =>
        decoder_writes_trailer_i <= '0';
        almost_full_flag_i       <= almost_full_i;
        data_out                 <= (others => '0');

      when l1_id =>
        data_out(31 downto 29) <= "001";
        data_out(28)           <= header_error_flag_i ;
        data_out(27 downto 16) <= (others => '0');
        data_out(15 downto  8) <= serial_data_reg(12 downto 5);

      when bc_id =>
        data_out( 7 downto  0) <= serial_data_reg(13 downto 6);
        data_out_write_i       <= '1';

      when hit     =>
        if (almost_full_flag_i = '0') then     --
          data_out_write_i       <= '1';
          data_out(31 downto 28) <= "1000";
          data_out(27 downto 24) <= fe_reg(3 downto 0);
          data_out(23 downto 16) <= serial_data_reg(13 downto 6);  -- TOT
          data_out(15 downto 13) <= (others => '0');
          data_out(12 downto  8) <= serial_data_reg(18 downto 14); -- COLUMN
          data_out( 7 downto  0) <= serial_data_reg(26 downto 19); -- ROW
        end if;
          
      when fe_flag =>
        if (almost_full_flag_i = '0') then
          data_out_write_i       <= '1';
          data_out(31 downto 28) <= "0000";
          data_out(27 downto 24) <= fe_reg(3 downto 0);
          data_out(23 downto 13) <= (others => '0');
          data_out(12 downto  8) <= "11110";
          data_out( 7 downto  0) <= serial_data_reg(13 downto 6);
        end if;

      when mcc_flag =>
        if (almost_full_flag_i = '0') then
          data_out_write_i       <= '1';
          data_out(31 downto 28) <= "0001";
          data_out(27 downto 24) <= fe_reg(3 downto 0);
          data_out(23 downto 21) <= (others => '0');
          data_out(20 downto  0) <= serial_data_reg(26 downto 6);
        end if;

      when trailer  =>
        data_out(31 downto 29)   <= "010";
        data_out(28)             <= trailer_error_flag_i;
        data_out(27)             <= almost_full_flag_i;
        data_out(26 downto  0)   <= (others => '0');
        data_out_write_i         <= '1';
        decoder_writes_trailer_i <= '1';

      when raw_data_header =>
        data_out(31 downto 29) <= "001";
        data_out(28)           <= header_error_flag_i ;
        data_out(27 downto 16) <= (others => '0');
        if (config_mode_in = '1') then
          data_out(15 downto  8) <= serial_data_reg(19 downto 12);
          data_out( 7 downto  0) <= X"CD";
        else
          data_out(15 downto 8) <= (others => '0');
          data_out( 7 downto 0) <= X"ED";
        end if;
        data_out_write_i     <= '1';

      when wr_raw_data =>
        if (almost_full_flag_i = '0') then
          data_out(31 downto 28) <= X"6";
          data_out(27 downto  0) <= serial_data_reg(27 downto 0);
          data_out_write_i       <= '1';
        end if;

      when others => null;
    end case;
  end if;
end process;

end rtl;