-------------------------------------------------------------------------------
-- Lawrence Berkeley National Labs
-- ATLAS ROD ELECTRONICS
-------------------------------------------------------------------------------
-- Filename:
--    link_formatter_d160.vhd
-- Description:
--    ROD Front End Decoder
--       Four   Channel
--       PIXEL  Version
--
-- This file contains the serial decoding state machine for the PIXEL front
-- end. You should refer to three documents to better understand how the
-- vhdl for the decoder works.
--
--    1) "MCC-D2 Specifications- Output Data Format" presents a state 
--    transition diagram that shows how to decode the MCC serial data. 
--    The VHDL logic for the PIXEL decoder is closely based on this diagram.
--
-- Note: There should also be numerous other documents describing the PIXEL
-- decoder and its output data stream format.
--
-- In general, the PIXEL decoder must be robust to errors (i.e. cannot
-- go into an undefined state regardless of input data). Therefore,
-- error handling is also important and this adds to the
-- complexity/amount of logic in the decoder.

-------------------------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
-------------------------------------------------------------------------------
entity link_formatter_d160 is
  port (
    clk_in                     : in  std_logic;
    rst_n_in                   : in  std_logic;
    almost_full_in             : in  std_logic;
    half_clk_err_mask_in       : in  std_logic;
    enable_in                  : in  std_logic;
    config_mode_in             : in  std_logic;
    link_in                    : in  std_logic_vector(3 downto 0);
    normal_not_raw_out         : out std_logic;
    data_strobe_out            : out std_logic;
    decoder_writes_trailer_out : out std_logic;
    decoded_data_out           : out std_logic_vector(31 downto 0);
    decoder_state_in_idle      : out std_logic
    );
end link_formatter_d160;
-------------------------------------------------------------------------------

architecture rtl of link_formatter_d160 is

-------------------------------------------------------------------------------
-- SIGNAL DECLARATIONS
-------------------------------------------------------------------------------
  type pixel_state_typedef is (
    idle,             -- 0x0001h
    store_id,         -- 0x0002h
    mcc_fe_test,      -- 0x0008h
    fe_num,           -- 0x0010h
    test,             -- 0x0020h
    hit,              -- 0x0040h   
    mcc_flag,         -- 0x0080h
    fe_flag,          -- 0x0100h
    raw_data_header,  -- 0x0200h
    trap_raw_data,    -- 
    wr_raw_data,      -- 
    wait_for_trailer, -- 0x0400h
    trailer           -- 0x0800h
    );
  signal dpar_state : pixel_state_typedef;

  signal almost_full_i        : std_logic;
  signal dpar_hit             : std_logic_vector(31 downto 0);
  signal start_write          : std_logic;
  signal wr_header            : std_logic;
  signal wr_trailer           : std_logic;
  signal serial_data_reg      : std_logic_vector(28 downto 0) ;
  signal fe_reg               : std_logic_vector( 3 downto 0);
  signal header_sh_reg        : std_logic_vector( 8 downto 0) ;
  signal header_detect_i      : std_logic;
  signal header_error_i       : std_logic;
  signal link_in_i            : std_logic_vector( 3 downto 0);
  signal trailer_detect_i     : std_logic;
  signal trailer_error_i      : std_logic;
  

begin
-------------------------------------------------------------------------------
-- Process Declaration
-------------------------------------------------------------------------------
input_register : process (
  clk_in, 
  rst_n_in
  )
begin
  if (rst_n_in = '0') then
    almost_full_i   <= '0' ;
    link_in_i       <= (others => '0');
--  serial_data_reg <= (others => '0');
  elsif (clk_in'event AND clk_in = '1') then
    almost_full_i   <= almost_full_in;
    if (enable_in = '1') then
      link_in_i <= link_in;
    else
      link_in_i <= (others => '0');
    end if;  
    serial_data_reg <= serial_data_reg(24 downto 0) & link_in_i;
  end if;
end process input_register;

-------------------------------------------------------------------------------
-- Process to detect Link Headers
find_header : process (
  clk_in, 
  rst_n_in
  )
begin  -- find header
  if (rst_n_in = '0') then
--  header_sh_reg  <= (others => '0');
    header_error_i <= '0';
  elsif (clk_in'event AND clk_in = '1') then
-- Correct header pattern is 11101.  A header can also have up to one bit 
-- of error in any location.         
    header_sh_reg <= header_sh_reg(4 downto 0) & serial_data_reg(11 downto 8);
    if (dpar_state = idle) then
      case header_sh_reg(8 downto 3) is
        when "011101" =>      -- "111101" is a possible hit pattern with an error 
          header_detect_i <= '1';
          header_error_i  <= header_sh_reg(8);

        when "111101" =>      -- "111101" is a possible hit pattern with an error 
          header_detect_i <= '1';
          header_error_i  <= NOT half_clk_err_mask_in; -- mask in half clock mode

        when "001101" =>
          header_detect_i <= '1';
          header_error_i  <= '1';

        when "010101" =>
          header_detect_i <= '1';
          header_error_i  <= '1';

        when "011001" =>
          header_detect_i <= '1';
          header_error_i  <= '1';

        when "011111" => -- This test requires at least 1 zero somewhere in the data stream
          header_detect_i <= NOT header_sh_reg(2);
          header_error_i  <= NOT header_sh_reg(2);

        when "011100" =>
          header_detect_i <= '1';
          header_error_i  <= '1';

        when others =>
          header_detect_i <= '0';
          header_error_i  <= '0';
      end case;
    else
      header_detect_i <= '0';
      header_error_i  <= '0';
    end if;
  end if;
end process find_header;
  
-------------------------------------------------------------------------------
-- Process to detect trailers only used if Formatter goes into raw data mode
find_trailer : process (
  clk_in, 
  rst_n_in
  )    
begin -- find_trailer
  if (rst_n_in = '0') then
    trailer_detect_i <= '0';
    trailer_error_i  <= '0';
  elsif (clk_in'event AND clk_in = '1') then
    if (serial_data_reg(21 downto 0) = "0000000000000000000000") then
      trailer_detect_i <= '1';
      trailer_error_i  <= '0';
--    elsif (serial_data_reg(21 downto 0) = "1111111111111111111111") then
--      trailer_detect_i <= '1';
--      trailer_error_i  <= NOT config_mode_in;
    else
      trailer_detect_i <= '0';
      trailer_error_i  <= '0';
    end if;
  end if;  
end process find_trailer;

-------------------------------------------------------------------------------
-- Serial to parallel decoder <==> 160MHz Input to ROD
pixel160_data_parser : process (
  rst_n_in,
  clk_in
  )
variable wait_count    : unsigned(4 downto 0);
variable sync_count_to : unsigned(2 downto 0);
variable sync_bit_loc  : std_logic_vector(3 downto 0);
begin
  if (rst_n_in = '0') then
    dpar_state	  <= idle;
    wait_count    := (others => '0');
    sync_count_to := (others => '0');
    sync_bit_loc  := (others => '0');
    start_write   <= '0';
    wr_header     <= '0';
    wr_trailer    <= '0';
    decoder_state_in_idle <= '1';
  elsif (clk_in'EVENT AND clk_in = '1') then
    start_write <= '0';
    wr_header   <= '0';
    wr_trailer  <= '0';
    case dpar_state is
      when idle =>
        sync_count_to := (others => '0');
        wait_count    := (others => '0');
        sync_bit_loc  := (others => '0');
        if (header_detect_i = '1') then  -- wake up on mci header detect
          dpar_state <= store_id;
        end if;
        decoder_state_in_idle <= '1';

      when store_id =>  -- store L1ID
        dpar_hit(31 downto 29) <= "001";
        dpar_hit(28)           <= header_error_i ;
        dpar_hit(27 downto 16) <= (others => '0');
        dpar_hit(15 downto  8) <= serial_data_reg(22 downto 15);
        if (serial_data_reg(14) = '1') then
          dpar_hit(7 downto 0) <= serial_data_reg(13 downto  6);
          wr_header    <= '1';
          dpar_state   <= mcc_fe_test;
        else
          dpar_hit(7 downto 0) <= (others => '0');
          dpar_state <= raw_data_header;
        end if;
        decoder_state_in_idle <= '0';
 
       when mcc_fe_test =>
        if (serial_data_reg(9 downto 5) = "11110") then
          dpar_state <= fe_num;
        else   ---if (serial_data_reg(9 downto 0) = "1000000000") then
          dpar_state <= wait_for_trailer;
        end if;
        sync_bit_loc  := "0001";
      
      when fe_num =>
        if    (sync_bit_loc = "0000") then fe_reg <= serial_data_reg( 7 downto 4);
        elsif (sync_bit_loc = "0001") then fe_reg <= serial_data_reg( 8 downto 5);
        elsif (sync_bit_loc = "0010") then fe_reg <= serial_data_reg( 9 downto 6);
        elsif (sync_bit_loc = "0100") then fe_reg <= serial_data_reg(10 downto 7);
        elsif (sync_bit_loc = "1000") then fe_reg <= serial_data_reg(11 downto 8);
        end if;
        dpar_state <= test;

      when test =>  -- test for next data type
        wait_count := (others => '0');
--  Test here for:  FE_FLAG  => "s11110ffffmmmm"
--                  MCC_FLAG => "s11111ffffffffmmmmmmmm"
--                  FE_NUM   => "s1110nnnn"
        case sync_bit_loc is 
          when "0000" =>
            if (serial_data_reg(7 downto 2) = "111110") then
              dpar_state <= fe_flag;
              sync_count_to := "010";
              sync_bit_loc  := "0010";
            elsif (serial_data_reg(7 downto 2) = "111111") then
              dpar_state <= mcc_flag;
              sync_count_to := "100";
              sync_bit_loc  := "0010";
            else
              dpar_state <= hit; 
              sync_count_to := "100";
              sync_bit_loc  := "0010";
            end if;
          when "0001" =>
            if (serial_data_reg(8 downto 3) = "111110") then
              dpar_state <= fe_flag;
              sync_count_to := "010";
              sync_bit_loc  := "0100";
            elsif (serial_data_reg(8 downto 3) = "111111") then
              dpar_state <= mcc_flag;
              sync_count_to := "100";
              sync_bit_loc  := "0100";
            elsif (serial_data_reg(8 downto 4) = "11110") then
              dpar_state <= fe_num;
              sync_count_to := "000";
              sync_bit_loc  := "0000";
            else
              dpar_state <= hit; 
              sync_count_to := "100";
              sync_bit_loc  := "0100";
            end if;
          when "0010" =>
            if (serial_data_reg(9 downto 4) = "111110") then
              dpar_state <= fe_flag;
              sync_count_to := "010";
              sync_bit_loc  := "1000";
            elsif (serial_data_reg(9 downto 4) = "111111") then
              dpar_state <= mcc_flag;
              sync_count_to := "100";
              sync_bit_loc  := "1000";
            elsif (serial_data_reg(9 downto 5) = "11110") then
              dpar_state <= fe_num;
              sync_count_to := "000";
              sync_bit_loc  := "0001";
            else
              dpar_state <= hit; 
              sync_count_to := "100";
              sync_bit_loc  := "1000";
            end if;
          when "0100" =>
            if (serial_data_reg(10 downto 5) = "111110") then
              dpar_state <= fe_flag;
              sync_count_to := "001";
              sync_bit_loc  := "0001";
            elsif (serial_data_reg(10 downto 5) = "111111") then
              dpar_state <= mcc_flag;
              sync_count_to := "011";
              sync_bit_loc  := "0001";
            elsif (serial_data_reg(10 downto 6) = "11110") then
              dpar_state <= fe_num;
              sync_count_to := "000";
              sync_bit_loc  := "0010";
            else
              dpar_state <= hit; 
              sync_count_to := "011";
              sync_bit_loc  := "0001";
            end if;
          when "1000" =>
            if (serial_data_reg(11 downto 6) = "111110") then
              dpar_state <= fe_flag;
              sync_count_to := "001";
              sync_bit_loc  := "0010";
            elsif (serial_data_reg(11 downto 6) = "111111") then
              dpar_state <= mcc_flag;
              sync_count_to := "011";
              sync_bit_loc  := "0010";
            elsif (serial_data_reg(11 downto 7) = "11110") then
              dpar_state <= fe_num;
              sync_count_to := "000";
              sync_bit_loc  := "0100";
            else
              dpar_state <= hit; 
              sync_count_to := "011";
              sync_bit_loc  := "0010";
            end if;
          when others => 
        end case;

      when hit =>
        if (sync_bit_loc = "0001") then
          if (wait_count = sync_count_to) then
            if (serial_data_reg(4) = '1') then
              dpar_hit(31 downto 28) <= "1000";
              dpar_hit(27 downto 24) <= fe_reg(3 downto 0);
              dpar_hit(23 downto 16) <= serial_data_reg(12 downto  5);  -- TOT
              dpar_hit(15 downto 13) <= (others => '0');
              dpar_hit(12 downto  8) <= serial_data_reg(17 downto 13);  -- COL
              dpar_hit( 7 downto  0) <= serial_data_reg(25 downto 18);  -- ROW
              dpar_state  <= test;
              start_write <= '1';
            elsif (serial_data_reg(25 downto 4) = "0000000000000000000000") then
              dpar_hit(31 downto 28) <= "0100";
              dpar_hit(27)           <= almost_full_i;
              dpar_hit(26 downto  0) <= (others => '0');
              wr_trailer             <= '1';
              dpar_state             <= idle;
            else 
              dpar_hit(31 downto 29) <= "011";
              dpar_hit(28 downto  0) <= serial_data_reg(28 downto 0);
              start_write <= '1';
              wait_count  := (others => '0');
              dpar_state  <= wait_for_trailer;
            end if;
          else
            wait_count := wait_count + 1;
          end if;
        elsif (sync_bit_loc = "0010") then
          if (wait_count = sync_count_to) then
            if (serial_data_reg(5) = '1') then
              dpar_hit(31 downto 28) <= "1000";
              dpar_hit(27 downto 24) <= fe_reg(3 downto 0);
              dpar_hit(23 downto 16) <= serial_data_reg(13 downto  6);  -- TOT
              dpar_hit(15 downto 13) <= (others => '0');
              dpar_hit(12 downto  8) <= serial_data_reg(18 downto 14);  -- COL
              dpar_hit( 7 downto  0) <= serial_data_reg(26 downto 19);  -- ROW
              dpar_state  <= test;
              start_write <= '1';
            elsif (serial_data_reg(26 downto 5) = "0000000000000000000000") then
              dpar_hit(31 downto 28) <= "0100";
              dpar_hit(27)           <= almost_full_i;
              dpar_hit(26 downto  0) <= (others => '0');
              wr_trailer             <= '1';
              dpar_state             <= idle;
            else 
              dpar_hit(31 downto 29) <= "011";
              dpar_hit(28 downto  0) <= serial_data_reg(28 downto 0);
              start_write <= '1';
              wait_count  := (others => '0');
              dpar_state  <= wait_for_trailer;
            end if;
          else
            wait_count := wait_count + 1;
          end if;
        elsif (sync_bit_loc = "0100") then
          if (wait_count = sync_count_to) then
            if (serial_data_reg(6) = '1') then
              dpar_hit(31 downto 28) <= "1000";
              dpar_hit(27 downto 24) <= fe_reg(3 downto 0);
              dpar_hit(23 downto 16) <= serial_data_reg(14 downto  7);  -- TOT
              dpar_hit(15 downto 13) <= (others => '0');
              dpar_hit(12 downto  8) <= serial_data_reg(19 downto 15);  -- COL
              dpar_hit( 7 downto  0) <= serial_data_reg(27 downto 20);  -- ROW
              dpar_state  <= test;
              start_write <= '1';
            elsif (serial_data_reg(27 downto 6) = "0000000000000000000000") then
              dpar_hit(31 downto 28) <= "0100";
              dpar_hit(27)           <= almost_full_i;
              dpar_hit(26 downto  0) <= (others => '0');
              wr_trailer             <= '1';
              dpar_state             <= idle;
            else 
              dpar_hit(31 downto 29) <= "011";
              dpar_hit(28 downto  0) <= serial_data_reg(28 downto 0);
              start_write <= '1';
              wait_count  := (others => '0');
              dpar_state  <= wait_for_trailer;
            end if;
          else
            wait_count := wait_count + 1;
          end if;
        elsif (sync_bit_loc = "1000") then
          if (wait_count = sync_count_to) then
            if (serial_data_reg(7) = '1') then
              dpar_hit(31 downto 28) <= "1000";
              dpar_hit(27 downto 24) <= fe_reg(3 downto 0);
              dpar_hit(23 downto 16) <= serial_data_reg(15 downto  8);  -- TOT
              dpar_hit(15 downto 13) <= (others => '0');
              dpar_hit(12 downto  8) <= serial_data_reg(20 downto 16);  -- COL
              dpar_hit( 7 downto  0) <= serial_data_reg(28 downto 21);  -- ROW
              dpar_state  <= test;
              start_write <= '1';
            elsif (serial_data_reg(28 downto 7) = "0000000000000000000000") then
              dpar_hit(31 downto 28) <= "0100";
              dpar_hit(27)           <= almost_full_i;
              dpar_hit(26 downto  0) <= (others => '0');
              wr_trailer             <= '1';
              dpar_state             <= idle;
            else 
              dpar_hit(31 downto 29) <= "011";
              dpar_hit(28 downto  0) <= serial_data_reg(28 downto 0);
              start_write <= '1';
              wait_count  := (others => '0');
              dpar_state  <= wait_for_trailer;
            end if;
          else
            wait_count := wait_count + 1;
          end if;
        end if;
 
      when fe_flag =>
        if (sync_bit_loc = "0001") then
          if (wait_count = sync_count_to) then
            if (serial_data_reg(4) = '1') then
              dpar_hit(31 downto 28) <= "0000";
              dpar_hit(27 downto 24) <= fe_reg(3 downto 0);
              dpar_hit(23 downto 13) <= (others => '0');
              dpar_hit(12 downto  0) <= serial_data_reg(17 downto 5);
              dpar_state             <= test;
              start_write            <= '1';
            else 
              dpar_hit(31 downto 29) <= "011";
              dpar_hit(28 downto  0) <= serial_data_reg(28 downto 0);
              start_write <= '1';
              wait_count  := (others => '0');
              dpar_state  <= wait_for_trailer;
            end if;
          else
            wait_count := wait_count + 1;
          end if;
        elsif (sync_bit_loc = "0010") then
          if (wait_count = sync_count_to) then
            if (serial_data_reg(5) = '1') then
              dpar_hit(31 downto 28) <= "0000";
              dpar_hit(27 downto 24) <= fe_reg(3 downto 0);
              dpar_hit(23 downto 13) <= (others => '0');
              dpar_hit(12 downto  0) <= serial_data_reg(18 downto 6);
              dpar_state             <= test;
              start_write            <= '1';
            else 
              dpar_hit(31 downto 29) <= "011";
              dpar_hit(28 downto  0) <= serial_data_reg(28 downto 0);
              start_write <= '1';
              wait_count  := (others => '0');
              dpar_state  <= wait_for_trailer;
            end if;
          else
            wait_count := wait_count + 1;
          end if;
        elsif (sync_bit_loc = "0100") then
          if (wait_count = sync_count_to) then
            if (serial_data_reg(6) = '1') then
              dpar_hit(31 downto 28) <= "0000";
              dpar_hit(27 downto 24) <= fe_reg(3 downto 0);
              dpar_hit(23 downto 13) <= (others => '0');
              dpar_hit(12 downto  0) <= serial_data_reg(19 downto 7);
              dpar_state             <= test;
              start_write            <= '1';
            else 
              dpar_hit(31 downto 29) <= "011";
              dpar_hit(28 downto  0) <= serial_data_reg(28 downto 0);
              start_write <= '1';
              wait_count  := (others => '0');
              dpar_state  <= wait_for_trailer;
            end if;
          else
            wait_count := wait_count + 1;
          end if;
        elsif (sync_bit_loc = "1000") then
          if (wait_count = sync_count_to) then
            if (serial_data_reg(7) = '1') then
              dpar_hit(31 downto 28) <= "0000";
              dpar_hit(27 downto 24) <= fe_reg(3 downto 0);
              dpar_hit(23 downto 13) <= (others => '0');
              dpar_hit(12 downto  0) <= serial_data_reg(20 downto 8);
              dpar_state             <= test;
              start_write            <= '1';
            else 
              dpar_hit(31 downto 29) <= "011";
              dpar_hit(28 downto  0) <= serial_data_reg(28 downto 0);
              start_write <= '1';
              wait_count  := (others => '0');
              dpar_state  <= wait_for_trailer;
            end if;
          else
            wait_count := wait_count + 1;
          end if;
        end if;
 
      when mcc_flag =>
        if (sync_bit_loc = "0001") then
          if (wait_count = sync_count_to) then
            if (serial_data_reg(4) = '1') then
              dpar_hit(31 downto 28) <= "0001";
              dpar_hit(27 downto 24) <= fe_reg(3 downto 0);
              dpar_hit(23 downto 21) <= (others => '0');
              dpar_hit(20 downto  0) <= serial_data_reg(25 downto 5);
              dpar_state  <= test;
              start_write <= '1';
            else 
              dpar_hit(31 downto 29) <= "011";
              dpar_hit(28 downto  0) <= serial_data_reg(28 downto 0);
              start_write <= '1';
              wait_count  := (others => '0');
              dpar_state  <= wait_for_trailer;
            end if;
          else
            wait_count := wait_count + 1;
          end if;
        elsif (sync_bit_loc = "0010") then
          if (wait_count = sync_count_to) then
            if (serial_data_reg(5) = '1') then
              dpar_hit(31 downto 28) <= "0001";
              dpar_hit(27 downto 24) <= fe_reg(3 downto 0);
              dpar_hit(23 downto 21) <= (others => '0');
              dpar_hit(20 downto  0) <= serial_data_reg(26 downto 6);
              dpar_state  <= test;
              start_write <= '1';
            else 
              dpar_hit(31 downto 29) <= "011";
              dpar_hit(28 downto  0) <= serial_data_reg(28 downto 0);
              start_write <= '1';
              wait_count  := (others => '0');
              dpar_state  <= wait_for_trailer;
            end if;
          else
            wait_count := wait_count + 1;
          end if;
        elsif (sync_bit_loc = "0100") then
          if (wait_count = sync_count_to) then
            if (serial_data_reg(6) = '1') then
              dpar_hit(31 downto 28) <= "0001";
              dpar_hit(27 downto 24) <= fe_reg(3 downto 0);
              dpar_hit(23 downto 21) <= (others => '0');
              dpar_hit(20 downto  0) <= serial_data_reg(27 downto 7);
              dpar_state  <= test;
              start_write <= '1';
            else 
              dpar_hit(31 downto 29) <= "011";
              dpar_hit(28 downto  0) <= serial_data_reg(28 downto 0);
              start_write <= '1';
              wait_count  := (others => '0');
              dpar_state  <= wait_for_trailer;
            end if;
          else
            wait_count := wait_count + 1;
          end if;
        elsif (sync_bit_loc = "1000") then
          if (wait_count = sync_count_to) then
            if (serial_data_reg(7) = '1') then
              dpar_hit(31 downto 28) <= "0001";
              dpar_hit(27 downto 24) <= fe_reg(3 downto 0);
              dpar_hit(23 downto 21) <= (others => '0');
              dpar_hit(20 downto  0) <= serial_data_reg(28 downto 8);
              dpar_state  <= test;
              start_write <= '1';
            else 
              dpar_hit(31 downto 29) <= "011";
              dpar_hit(28 downto  0) <= serial_data_reg(28 downto 0);
              start_write <= '1';
              wait_count  := (others => '0');
              dpar_state  <= wait_for_trailer;
            end if;
          else
            wait_count := wait_count + 1;
          end if;
        end if;

      when raw_data_header => 
        start_write <= '1';
        dpar_state  <= wait_for_trailer;

      when wait_for_trailer =>
        if (trailer_detect_i = '1') then
          dpar_state <= trailer;
        end if;

      when trailer =>
        dpar_hit(31 downto 29) <= "010";
        dpar_hit(28)           <= trailer_error_i;
        dpar_hit(27)           <= almost_full_i;
        dpar_hit(26 downto  0) <= (others => '0');
        wr_trailer             <= '1';
        dpar_state             <= idle;

      when others =>
        dpar_state <= idle;
    end case;
  end if;
end process pixel160_data_parser;

-------------------------------------------------------------------------------
-- purpose: This process looks at the current state and outputs the data
-- words depending on which state and which bit is being shifted in
wr_output_data : process (
  clk_in, 
  rst_n_in
  )
begin  -- process
  if (rst_n_in = '0') then
    data_strobe_out            <= '0';
    decoder_writes_trailer_out <= '0';
  elsif (clk_in'event AND clk_in = '1') then
    data_strobe_out            <= (start_write AND NOT almost_full_i) OR 
                                  wr_header OR 
                                  wr_trailer;
    decoder_writes_trailer_out <= wr_trailer;
  end if;
end process;
decoded_data_out   <= dpar_hit;
normal_not_raw_out <= '1';

end rtl;

