-------------------------------------------------------------------------------
--
-- Particle Physics Detector Electronics Support 
-- University of Wisconsin
-- Lawrence Berkeley National Laboratory (c) 1999
-- ReadOutDriver Electronics
-------------------------------------------------------------------------------
--
-- Filename: trigger_counter_token_generator.vhd
-- Description:
--
-------------------------------------------------------------------------------
-- Structure: 
--  Multiple state machines in a flat topology/hierarchy which communicate
--   with single wire signals.  
-------------------------------------------------------------------------------
-- Timing:
--    *The Engine FPGA runs at 40MHz => clk40
--    *The timing structure of all engine code will be:
--      1.  Perform all logic operations on positive edge of clk40
--      2.  Perform all memory operations on positive edge of clk40
-------------------------------------------------------------------------------
-- Author:         Mark L. Nagel
-- Board Engineer: John Joseph 
-- History:
--    Friday 19 May 2000: 1st version
--    Modified : 09 Dec 2002  - JMJ
-------------------------------------------------------------------------------

-------------------------------------------------------------------------------
-- LIBRARY INCLUDES
-------------------------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;  -- needed for logic operations
use IEEE.std_logic_arith.all; -- needed for +/- operations

-------------------------------------------------------------------------------
-- PORT DECLARATION
-------------------------------------------------------------------------------

entity tctg is  -- Trigger Counter/Token Generator
  port(
    clk_in              : in  std_logic;
    rst_n_in            : in  std_logic;
    master_not_slave_in : in  std_logic;
    roc_ready_in        : in  std_logic;
    l1_trigger_pulse_in : in  std_logic;
    token_in            : in  std_logic;
    token_out           : out std_logic;
    trigger_count_out   : out std_logic_vector(9 downto 0)
    );
end tctg;

architecture rtl of tctg is

-------------------------------------------------------------------------------
-- SIGNAL DECLARATION
-------------------------------------------------------------------------------

type states is (
  reset, 
  idle, 
  issue_token, 
  wait_for_token
  );
signal pres_state : states;

signal trigger_count_i       : unsigned(8 downto 0);
signal token_out_i           : std_logic;
signal token_in_i            : std_logic;
signal l1_trigger_pulse_i    : std_logic;
signal trigger_count_error_i : std_logic;

-------------------------------------------------------------------------------
-- PROCESS DECLARATION
-------------------------------------------------------------------------------
begin
-------------------------------------------------------------------------------
-- Signal Connections
-------------------------------------------------------------------------------

trigger_count_out <= trigger_count_error_i & std_logic_vector(trigger_count_i);
token_out         <= token_out_i;
--token_out         <= ( (token_out_i AND     master_not_slave_in) OR
--                       (token_in_i  AND NOT master_not_slave_in) );

-------------------------------------------------------------------------------
-- Input Signal Register
-------------------------------------------------------------------------------
input_register : process (
  rst_n_in, 
  clk_in,
  token_in, 
  l1_trigger_pulse_in
  )
begin
  if (rst_n_in = '0' ) then
    token_in_i         <= '0';
    l1_trigger_pulse_i <= '0';
  elsif (clk_in'event AND clk_in = '1') then
    token_in_i         <= token_in;
    l1_trigger_pulse_i <= l1_trigger_pulse_in;
  end if;
end process input_register;

-------------------------------------------------------------------------------
-- Trigger Counter
-------------------------------------------------------------------------------
trigger_counter : process (
  rst_n_in, 
  clk_in
  )
begin
  if (rst_n_in = '0' ) then
    trigger_count_i <= (others => '0');
    trigger_count_error_i <= '0';
  elsif (clk_in'event AND clk_in = '1') then
    if (master_not_slave_in = '1') then
      if (l1_trigger_pulse_i = '1' AND token_in_i = '0') then
        trigger_count_i <= trigger_count_i + 1;
      elsif (l1_trigger_pulse_i = '0' AND token_in_i = '1' AND trigger_count_i > 0) then
        trigger_count_i <= trigger_count_i - 1;
      end if;
      if (trigger_count_i = 511 AND l1_trigger_pulse_i = '1' AND token_in_i = '0') then
        trigger_count_error_i <= '1';
      end if;
    else
      trigger_count_i <= (others => '0');
      trigger_count_error_i <= '0';
    end if;
  end if;
end process trigger_counter;

-------------------------------------------------------------------------------
-- Token Generator
-------------------------------------------------------------------------------
token_generator : process (
  rst_n_in, 
  clk_in,
  roc_ready_in, 
  token_in_i,
  trigger_count_i
  )
begin
  if (rst_n_in = '0' ) then
    token_out_i <= '0';
    pres_state  <= reset; 
  elsif (clk_in'event AND clk_in = '1') then
    if (master_not_slave_in = '1') then
      case pres_state is
        when reset =>
          token_out_i <= '0';
          pres_state  <= idle; 
  
        when idle =>
          token_out_i <= '0';
          if (trigger_count_i > 0 AND roc_ready_in = '1') then
            pres_state <= issue_token;
          end if; 
  
        when issue_token =>
          token_out_i <= '1';  -- Issue the token
          pres_state  <= wait_for_token;
  
        when wait_for_token => 
          token_out_i <= '0';  -- Clear token
          if (token_in_i = '1') then 
            pres_state  <= reset;
          end if; 
      end case;
    else  --  master_not_slave_in = '0'
      case pres_state is
        when reset =>
          token_out_i <= '0';
          pres_state  <= idle; 
  
        when idle =>
          token_out_i <= '0';
          if (token_in_i = '1') then 
            pres_state <= issue_token;
          end if; 
  
        when issue_token =>
          if (roc_ready_in = '1') then
            token_out_i <= '1';  -- Issue the token
            pres_state  <= wait_for_token;
          end if;
  
        when wait_for_token => 
          token_out_i <= '0';  -- Clear token
          pres_state  <= reset;
      end case;
    end if;
  end if;
end process;

end rtl; -- end code for tctg
