--------------------------------------------------------------------------
--
-- Particle Physics Detector Electronics Support 
-- University of Wisconsin
-- Lawrence Berkeley National Laboratory (c) 2000
-- ReadOutDriver Electronics
--
--------------------------------------------------------------------------
--
-- Filename: 512x8fifo.vhd                  
-- Description:
--  Input registered and Output registered fifo.     
--
--------------------------------------------------------------------------
-- Structure: 
--  Multiple state machines in a flat topology/hierarchy which communicate
--   with single wire signals.  
--------------------------------------------------------------------------
-- Timing:
--    *The Engine FPGA runs at 40MHz => clk
--    *The timing structure of all engine code will be:
--      1.  Perform all logic operations on positive edge of clk40
--      2.  Perform all memory operations on positive edge of clk40
--------------------------------------------------------------------------
-- Author: Mark L. Nagel
-- Board Engineer: John M. Joseph
-- History:
--    23 February 2000 - MLN first version : requirements/functionality finally
--                       understood and sufficiently documented to proceed 
--                       with coding.
-- Notes: 
--
--
--------------------------------------------------------------------------

--------------------------------------------------------------------------
-- LIBRARY INCLUDES
--------------------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;  -- needed for logic operations
use IEEE.std_logic_arith.all; -- needed for +/- operations

--------------------------------------------------------------------------
-- PORT DECLARATION
--------------------------------------------------------------------------
entity fifo_512x24 is
  port(
    clk_in              : in  std_logic; -- clk40 input
    rst_n_in            : in  std_logic; -- negative logic asynchronous global reset
    flush_fifo_in       : in  std_logic; -- flush fifo
    wren_in             : in  std_logic; -- enable input register
    rden_in             : in  std_logic; -- enable output register
    fifo_in             : in  std_logic_vector(23 downto 0); --
    fifo_out            : out std_logic_vector(23 downto 0); --
    full_flag_out       : out std_logic; --
    empty_flag_out      : out std_logic; --
    occupancy_count_out : out unsigned(8 downto 0)   --
   );
end fifo_512x24;

architecture rtl of fifo_512x24 is
--------------------------------------------------------------------------
-- SIGNAL DECLARATION
--------------------------------------------------------------------------

signal wen_i        : std_logic;
signal wr_addr      : unsigned(8 downto 0);
signal ren_i        : std_logic;
signal rd_addr      : unsigned(8 downto 0);
signal fifo_dout_i  : std_logic_vector(23 downto 0);
signal occ_count    : unsigned(8 downto 0);
signal fifo_full_i  : std_logic;
signal fifo_empty_i : std_logic;
signal logic0       : std_logic;
signal logic1       : std_logic;

--------------------------------------------------------------------------
-- COMPONENT DECLARATION
--------------------------------------------------------------------------

component dpram_512x8
  port (
    clk_in           : in  std_logic; -- clk40 input
    rst_n_in         : in  std_logic; -- asynchronous global reset
    dprm_writein_bus : in  std_logic_vector(7 downto 0);
    writein_addr     : in  unsigned(8 downto 0); -- 
    readout_addr     : in  unsigned(8 downto 0); --
    write_strobe     : in  std_logic; --
    read_strobe      : in  std_logic; --
    dprm_readout_bus : out std_logic_vector(7 downto 0)
    );
end component;

--------------------------------------------------------------------------
-- COMPONENT INSTANTIATION
--------------------------------------------------------------------------
begin

U1 : dpram_512x8
  port map (
    clk_in           => clk_in,
    rst_n_in         => rst_n_in,
    dprm_writein_bus => fifo_in(7 downto 0),
    writein_addr     => wr_addr,
    readout_addr     => rd_addr,
    write_strobe     => wren_in,
    read_strobe      => logic1,
    dprm_readout_bus => fifo_dout_i(7 downto 0)
    );

U2 : dpram_512x8
  port map (
    clk_in           => clk_in,
    rst_n_in         => rst_n_in,
    dprm_writein_bus => fifo_in(15 downto 8),
    writein_addr     => wr_addr,
    readout_addr     => rd_addr,
    write_strobe     => wren_in,
    read_strobe      => logic1,
    dprm_readout_bus => fifo_dout_i(15 downto 8)
    );

U3 : dpram_512x8
  port map (
    clk_in           => clk_in,
    rst_n_in         => rst_n_in,
    dprm_writein_bus => fifo_in(23 downto 16),
    writein_addr     => wr_addr,
    readout_addr     => rd_addr,
    write_strobe     => wren_in,
    read_strobe      => logic1,
    dprm_readout_bus => fifo_dout_i(23 downto 16)
    );

--------------------------------------------------------------------------
-- PROCESS DECLARATION
--------------------------------------------------------------------------

logic0 <= '0';
logic1 <= '1';

full_flag_out  <= fifo_full_i;
empty_flag_out <= fifo_empty_i;

occupancy_count_out <= occ_count;

-- delay wren to ram block by 1 clk
-- read/write enable delay register
wen_delay : process (
  rst_n_in, 
  clk_in, 
  wren_in
  )
begin
  if (rst_n_in = '0') then
    wen_i <= '0';
  elsif (clk_in'event AND clk_in = '1') then
    wen_i <= wren_in AND NOT fifo_full_i;
  end if;
end process wen_delay;

-- input  address counter
input_counter : process (
  rst_n_in, 
  clk_in, 
  flush_fifo_in,
  wen_i
  )
begin
  if (rst_n_in = '0') then
    wr_addr <= (others => '0');
  elsif (clk_in'event and clk_in = '1') then
    if (flush_fifo_in = '1') then
      wr_addr <= (others => '0'); 
    elsif (wen_i = '1') then
      wr_addr <= wr_addr + 1;
    end if;
  end if;
end process input_counter;

-- output  address counter
output_register : process (
  rst_n_in, 
  clk_in, 
  flush_fifo_in,
  rden_in
  )
begin
  if (rst_n_in = '0') then
    fifo_out <= (others => '0');
  elsif (clk_in'event and clk_in = '1') then
    if (flush_fifo_in = '1') then
      fifo_out <= (others => '0');
    elsif (rden_in = '1')  then
      fifo_out <= fifo_dout_i; 
    end if;
  end if;
end process output_register;

-- output  address counter
output_counter : process (
  rst_n_in, 
  clk_in, 
  flush_fifo_in,
  rden_in
  )
begin
  if (rst_n_in = '0') then
    rd_addr <= (others => '0');
  elsif (clk_in'event and clk_in = '1') then
    if (flush_fifo_in = '1') then
      rd_addr <= (others => '0');
    elsif (rden_in = '1' AND rd_addr = wr_addr)  then
      rd_addr <= rd_addr; 
    elsif (rden_in = '1') then
      rd_addr <= rd_addr + 1;
    end if;
  end if;
end process output_counter;


-- fifo full/empty flag comparator  
empty_flag_comparator : process (
  rst_n_in,
  clk_in,
  occ_count
  )
begin
  if (rst_n_in = '0') then
    fifo_empty_i <= '1';
  elsif (clk_in'event and clk_in = '1') then
    if (occ_count = 0 OR flush_fifo_in = '1') then 
      fifo_empty_i <= '1';
    else
      fifo_empty_i <= '0';
    end if;
  end if;
end process empty_flag_comparator;

-- fifo full/empty flag comparator  
full_flag_comparator : process (
  rst_n_in,
  occ_count,
  flush_fifo_in
  )
begin
  if (rst_n_in = '0' OR flush_fifo_in = '1') then
    fifo_full_i  <= '0';
  else
    if (occ_count = 511) then 
      fifo_full_i <= '1';
    else
      fifo_full_i <= '0';
    end if;
  end if;
end process full_flag_comparator;

-- fifo 32bit word occupancy counter
occupancy_counter : process (
  rst_n_in, 
  clk_in, 
  flush_fifo_in
  )
begin
  if (rst_n_in = '0') then
    occ_count <= (others => '0');
  elsif (clk_in'event and clk_in = '1') then
    if (flush_fifo_in = '1') then
      occ_count  <= (others => '0');
    elsif (wen_i = '1' AND rden_in = '1') then
      occ_count <= occ_count;
    elsif (wen_i = '1' AND occ_count < 511) then
      occ_count <= occ_count + 1;
    elsif (rden_in = '1' AND occ_count > 0) then
      occ_count <= occ_count - 1;
    elsif (wen_i = '1' AND occ_count = 511) then
      occ_count <= occ_count;
    elsif (rden_in = '1' AND occ_count = 0) then
      occ_count <= occ_count;
    end if;
  end if;
end process occupancy_counter;

end rtl;
