-------------------------------------------------------------------------------
-- Particle Physics Detector Electronics Support 
-- University of Wisconsin
-- Lawrence Berkeley National Laboratory (c) 2003
-- ReadOutDriver Electronics
-------------------------------------------------------------------------------
-- Filename: 2048x32fifo.vhd                  
-- Description:
--  Input Link FIFO and control.  Registered inputs and outputs and complex
--  event data output control.  Monitoring of L1ID and MCC skipped event 
--  Header/Trailer insertion.
-------------------------------------------------------------------------------
-- Structure: 
--  Multiple state machines in a flat topology/hierarchy which communicate
--   with single wire signals.  
-------------------------------------------------------------------------------
-- Timing:
--    *The Engine FPGA runs at 40MHz => clk
--    *The timing structure of all engine code will be:
--      1.  Perform all logic operations on positive edge of clk40
--      2.  Perform all memory operations on positive edge of clk40
-------------------------------------------------------------------------------
-- History:
--   23 February 2000 - JMJ first version : 
-- Notes: 
--   06 October 2008  
--     MCC Empty Events (MCC_EE) - Pixel Only
--       A value in the upper nibble of the L1ID field of an event from the MCC
--       indicates that the FE Module L1ID FIFO overflowed, and that there are 
--       missing events int he data stream.  The ReadOut Controller is required
--       to insert the appropriate number of events to maintain synchronization
--       with the TTC system.  The FIFO Readout Controller will handle this type
--       of error iSIM:Simulation 1n the following way:
--         1.)  After detecting a header, the skipped event count (7:4) and 
--              current L1ID (3:0) will be latched into holding registers.
--              if the skipped event count is greater than 0, the field will
--              be set to 0 in the current header that is in process, and the 
--              skipped event count will be transferred into the empty event 
--              counter register. 
--         2.)  Output data format of an inserted skipped event:
--                Header => 0x2000SLBC 
--                               (S >> # skipped triggers 1 to 15)
--                               (L >> L1ID value) 
--                Trailer => 0x40000000
--
--      Add functionality to handle data corruption form the MCC.  Following
--      an MCC Skipped Event, check the next header.  If it has a MCC Skipped
--      count or the correct L1ID, leave it alone.  If has a small L1ID and
--      the MCC skipped flag is 0, throw it out.
-- 
-------------------------------------------------------------------------------
-- LIBRARY INCLUDES
-------------------------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;  -- needed for logic operations
use IEEE.std_logic_arith.all; -- needed for +/- operations
use IEEE.std_logic_unsigned.all; -- needed for +/- operations
-------------------------------------------------------------------------------
-- PORT DECLARATION
-------------------------------------------------------------------------------
entity fifo_2048x32 is
  port(
    clk_in            : in  std_logic;
    rst_n_in          : in  std_logic;
    link_enable_in    : in  std_logic;
    config_mode_in    : in  std_logic;
    num_accepts_in    : in  std_logic_vector( 3 downto 0); --
    to_error_in       : in  std_logic;
    wren_in           : in  std_logic;
    rden_in           : in  std_logic;
    fifo_pause_in     : in  std_logic;
    fifo_in           : in  std_logic_vector(31 downto 0); --
    fifo_out          : out std_logic_vector(31 downto 0); --
    data_valid_out    : out std_logic; --
    fifo_ef_out       : out std_logic;
    fifo_ff_out       : out std_logic;
    occupancy_out     : out unsigned(10 downto 0);
    fifo_eoe_word_out : out std_logic;
    num_accepts_out   : out std_logic_vector( 3 downto 0);
    ecr_in            : in  std_logic;
    tim_l1id_in       : in  std_logic_vector( 3 DOWNTO 0);
    last_l1id_in      : in  std_logic_vector(23 DOWNTO 0);
    dbg_count_ctrl_in : in  std_logic_vector( 1 downto 0);
    dproc_count_en_in : in  std_logic;
    ecr_count_en_in   : in  std_logic;
    resync_count_out  : out std_logic_vector(31 DOWNTO 0);
    hitword_count_out : out std_logic_vector(31 DOWNTO 0);
    event_count_out   : out std_logic_vector(31 DOWNTO 0);
    error_count_out   : out std_logic_vector(31 DOWNTO 0);
    dproc_count_out   : out std_logic_vector(31 DOWNTO 0);
    clk_count_out     : out std_logic_vector(31 DOWNTO 0)
    );
end fifo_2048x32;

architecture rtl of fifo_2048x32 is
-------------------------------------------------------------------------------
-- SIGNAL DECLARATION
-------------------------------------------------------------------------------

signal fifo_din_i      : std_logic_vector(31 downto 0);
signal fifo_dout_i     : std_logic_vector(31 downto 0);
signal fifo_wen_i      : std_logic;
signal fifo_clr_i      : std_logic;
signal fifo_wr_i       : std_logic;
signal fifo_rd_i       : std_logic;
signal inc_wr_addr_i   : std_logic;
signal fifo_wr_addr    : unsigned(10 downto 0);
signal fifo_rd_addr    : unsigned(10 downto 0);
signal occupancy_count : unsigned(10 downto 0);
signal fifo_full_i     : std_logic;
signal fifo_empty_i    : std_logic;
signal data_valid_i    : std_logic;

type fifo_readout_states is (
  idle,
  play,
  check_next_word,
  send_mcc_ee,
  done
  );
signal fifo_readout : fifo_readout_states;

type mcc_empty_event_states is (
  idle,
  send_eeh,
  send_eet,
  check_next_word,
  done
  );
signal mcc_empty_event : mcc_empty_event_states;

signal num_accepts_i  : std_logic_vector(3 downto 0);
signal mcc_ee_count_i : std_logic_vector(4 downto 0);
signal mcc_ee_code_i  : std_logic_vector(4 downto 0);
signal pres_l1id_i    : std_logic_vector(3 downto 0);
signal pres_bcid_i    : std_logic_vector(7 downto 0);
signal ro_status      : std_logic_vector(3 downto 0);

signal l1id_check_i   : std_logic_vector(23 downto 0);

signal clr_next_mem_loc_i : std_logic;
signal mcc_ee_overflow : std_logic;

signal wait_delay      : unsigned( 1 downto 0);

signal resync_count_i  : unsigned(31 DOWNTO 0);
signal hitword_count_i : unsigned(31 DOWNTO 0);
signal event_count_i   : unsigned(31 DOWNTO 0);
signal error_count_i   : unsigned(31 DOWNTO 0);
signal dproc_count_i   : unsigned(31 DOWNTO 0);
signal clk_count_i     : unsigned(31 DOWNTO 0);

-------------------------------------------------------------------------------
-- COMPONENT DECLARATION
-------------------------------------------------------------------------------

component dpram_2048x2
  port(
    clk_in           : in  std_logic;
    rst_n_in         : in  std_logic;
    dprm_writein_bus : in  std_logic_vector(1 downto 0);
    writein_addr     : in  unsigned(10 downto 0); -- 
    readout_addr     : in  unsigned(10 downto 0); --
    write_strobe     : in  std_logic; --
    read_strobe      : in  std_logic; --
    dprm_readout_bus : out std_logic_vector(1 downto 0)
    );
end component;

begin
-------------------------------------------------------------------------------
-- COMPONENT INSTANTIATION
-------------------------------------------------------------------------------

fb : for i in 0 to 15 generate
  U1 : dpram_2048x2
    port map (
      clk_in           => clk_in,
      rst_n_in         => rst_n_in,
      dprm_writein_bus => fifo_din_i(2*i+1 downto 2*i),
      writein_addr     => fifo_wr_addr(10 downto 0),
      readout_addr     => fifo_rd_addr(10 downto 0),
      write_strobe     => fifo_wr_i,
      read_strobe      => fifo_rd_i,
      dprm_readout_bus => fifo_dout_i(2*i+1 downto 2*i)
      );
end generate;

-------------------------------------------------------------------------------
-- SIGNALS
-------------------------------------------------------------------------------
fifo_wr_i       <= fifo_wen_i OR clr_next_mem_loc_i;
fifo_rd_i       <= '1';
fifo_ef_out     <= fifo_empty_i;
fifo_ff_out     <= fifo_full_i;
occupancy_out   <= occupancy_count;
num_accepts_out <= num_accepts_i;
-------------------------------------------------------------------------------
-- PROCESS DECLARATION
-------------------------------------------------------------------------------
-- Debug Counters
resync_count_out  <= std_logic_vector(resync_count_i);
hitword_count_out <= std_logic_vector(hitword_count_i);
event_count_out   <= std_logic_vector(event_count_i);
error_count_out   <= std_logic_vector(error_count_i);
dproc_count_out   <= std_logic_vector(dproc_count_i);
clk_count_out     <= std_logic_vector(clk_count_i);

debug_counters_gen : process (
  rst_n_in, 
  clk_in
  )
begin
  if (rst_n_in = '0') then
    clk_count_i     <= (others => '0');
    dproc_count_i   <= (others => '0');
    resync_count_i  <= (others => '0');
    hitword_count_i <= (others => '0');
    event_count_i   <= (others => '0');
    error_count_i   <= (others => '0');
  elsif (clk_in'event AND clk_in = '1') then
    if (dbg_count_ctrl_in(1) = '1') then
      clk_count_i     <= (others => '0');
      dproc_count_i   <= (others => '0');
      resync_count_i  <= (others => '0');
      hitword_count_i <= (others => '0');
      event_count_i   <= (others => '0');
      error_count_i   <= (others => '0');
    elsif (dbg_count_ctrl_in(0) = '1') then
      clk_count_i   <= clk_count_i + '1';
      dproc_count_i <= dproc_count_i + dproc_count_en_in;
      
-- ecr counter
      resync_count_i <= resync_count_i + ecr_count_en_in;

--  when hit      => data_out(31 downto 28) <= "1000";
      if (fifo_wen_i = '1' AND fifo_din_i(31 downto 28) = X"8") then
        hitword_count_i <= hitword_count_i + '1';
      end if;

--  when header   => data_out(31 downto 29) <= "001" ;
--  when trailer  => data_out(31 downto 29) <= "010" ;
      if (fifo_wen_i = '1' AND fifo_din_i(31 downto 29) = "001") then
        event_count_i <= event_count_i + '1';
      end if;

--  when fe_flag  => data_out(31 downto 28) <= "0000";
--  when mcc_flag => data_out(31 downto 28) <= "0001";
      if (fifo_wen_i = '1' AND (fifo_din_i(31 downto 28) = "0000" OR fifo_din_i(31 downto 28) = "0001")) then
        error_count_i <= error_count_i + '1';
      end if;

--  when raw_data => data_out(31 downto 28) <= "0110";
    end if;
  end if;
end process debug_counters_gen;


-- Select the Write Enable used by the RAM Block.
-- If both links are active, each FIFO is 256x32
-- If one link is masked, the FIFO is 512x32
--  Following the FIFO write of event data, the WR Address is incremented by 1,
--  and the memory location is cleared of old data.  This is required so that
--  an old trailer will not be detected by the Readout Controller.
--  
wen_select : process (
  rst_n_in, 
  clk_in, 
  link_enable_in
  )
begin
  if (rst_n_in = '0') then
    fifo_wen_i         <= '0';
    inc_wr_addr_i      <= '0';
    clr_next_mem_loc_i <= '0';
    fifo_clr_i         <= '0';
  elsif (clk_in'event AND clk_in = '1') then
    if (link_enable_in = '1') then
      fifo_wen_i         <= wren_in AND NOT fifo_full_i;
      inc_wr_addr_i      <= wren_in AND NOT fifo_full_i;
      clr_next_mem_loc_i <= inc_wr_addr_i AND NOT fifo_full_i;
    else
      fifo_wen_i         <= '0';
      inc_wr_addr_i      <= '0';
      clr_next_mem_loc_i <= '0';
    end if;
  end if;
end process wen_select;

-------------------------------------------------------------------------------
-- input address counter
input_counters : process (
  rst_n_in, 
  clk_in,
  link_enable_in
  )
begin
  if (rst_n_in = '0') then
    fifo_wr_addr <= (others => '0');
  elsif (clk_in'event AND clk_in = '1') then
    if (link_enable_in = '0' OR ecr_in = '1') then -- hold link in reset
      fifo_wr_addr <= (others => '0');
    else
      if (inc_wr_addr_i = '1') then
        fifo_wr_addr <= fifo_wr_addr + 1;
      end if;
    end if;
  end if;
end process input_counters;

-------------------------------------------------------------------------------
input_mux : process (
  rst_n_in, 
  clk_in,
  link_enable_in,
  fifo_in
  )
begin
  if (rst_n_in = '0') then
    fifo_din_i <= (others => '0');
  elsif (clk_in'event AND clk_in = '1') then
    if (wren_in = '1') then 
      fifo_din_i <= fifo_in;
    end if;

--  After the Event Data is written to the FIFO, clear the data register for
--  the Next Location Clear operation    
    if (fifo_wen_i = '1') then
      fifo_din_i <= (others => '0');
    end if;
  end if;
end process;
      
-------------------------------------------------------------------------------
output_mux : process (
  rst_n_in,
  clk_in,
  link_enable_in 
  )
begin
  if (rst_n_in = '0') then
    fifo_out <= (others => '0');
    data_valid_out <= '0';
  elsif (clk_in'event AND clk_in = '1') then
    data_valid_out <= data_valid_i;
    if (link_enable_in = '1') then
      if (mcc_empty_event = send_eeh) then  -- MCC Empty Event Header
        fifo_out(31 downto 16) <= X"2000";
        fifo_out(15 downto 12) <= mcc_ee_code_i(3 downto 0);
        fifo_out(11 downto  8) <= pres_l1id_i;
        fifo_out( 7 downto  0) <= pres_bcid_i + 1 ;
      elsif (mcc_empty_event = send_eet) then  -- MCC Empty Event Trailer
        fifo_out <= X"40000000";
      else  -- for all other states
        if (fifo_dout_i(31 downto 29) = "001") then  -- Header Data Output
          if (config_mode_in = '0') then
            fifo_out <= fifo_dout_i(31 downto 16) & "0000" & fifo_dout_i(11 downto 0);
          else
            fifo_out <= fifo_dout_i;
          end if;
        else  -- Hits, Trailers and Error words
          fifo_out <= fifo_dout_i;
        end if;
      end if;          
    else
      fifo_out <= (others => '0'); 
    end if;
  end if;
end process output_mux;
      
-------------------------------------------------------------------------------
-- output address counter
output_counter : process (
  rst_n_in, 
  clk_in
  )
begin
  if (rst_n_in = '0') then
    fifo_rd_addr  <= (others => '0');
    data_valid_i  <= '0';
    num_accepts_i <= (others => '0');
    mcc_ee_count_i <= (others => '0');
    mcc_ee_code_i <= (others => '0');
    pres_l1id_i <= (others => '0');
    l1id_check_i <= X"000001";
    pres_bcid_i <= (others => '0');
    fifo_eoe_word_out <= '0';
    fifo_readout  <= idle;
    mcc_empty_event <= idle;
    ro_status <= (others => '0');
    wait_delay <= (others => '0');
    mcc_ee_overflow <= '0';
  elsif (clk_in'event and clk_in = '1') then
    if (link_enable_in = '0' OR ecr_in = '1') then -- hold link in reset
      fifo_rd_addr  <= (others => '0');
      data_valid_i  <= '0';
      num_accepts_i <= (others => '0');
      mcc_ee_count_i <= (others => '0');
      mcc_ee_code_i <= (others => '0');
      pres_l1id_i <= (others => '0');
      l1id_check_i <= X"000001";
      pres_bcid_i <= (others => '0');
      fifo_eoe_word_out <= '0';
      fifo_readout  <= idle;
      mcc_empty_event <= idle;
      ro_status <= (others => '0');
      wait_delay <= (others => '0');
      mcc_ee_overflow <= '0';
    else
---- Clear State Machine when ECR arrives
--      if (ecr_in = '1') then
--        l1id_check_i <= X"000001";
--        mcc_ee_code_i <= (others => '0');
--        mcc_ee_count_i <= (others => '0');
--        mcc_ee_overflow <= '0';
--        num_accepts_i <= (others => '0');
--        fifo_readout  <= idle;
--        mcc_empty_event <= idle;
--      end if;  
      case fifo_readout is
        when idle => 
          data_valid_i <= '0';
          fifo_eoe_word_out <= '0';
          pres_bcid_i  <= (others => '0');
          wait_delay <= (others => '0');
          if (mcc_ee_count_i > 14) then
            mcc_ee_code_i <= "01111";
          else
            mcc_ee_code_i <= mcc_ee_count_i;
          end if;
          if (fifo_pause_in = '0') then
            if (rden_in = '1') then
              if (mcc_ee_count_i /= 0) then
                fifo_readout <= send_mcc_ee;
                pres_l1id_i  <= pres_l1id_i + '1';
              elsif (fifo_empty_i = '0') then
                data_valid_i <= '1';
                fifo_readout <= play;
                fifo_rd_addr <= fifo_rd_addr + 1;
              end if;
            end if;
          end if;

        when play =>
          if (to_error_in = '1') then
            fifo_readout <= done;
          elsif (fifo_empty_i = '0') then
            if (rden_in = '1' AND fifo_rd_addr /= fifo_wr_addr) then 
              if (num_accepts_in = "0000" OR num_accepts_i = num_accepts_in) then
                if (fifo_dout_i(31 downto 29) = "010") then
                  data_valid_i <= '0';
                  fifo_eoe_word_out <= '1';
                  fifo_readout <= done;
                elsif (fifo_pause_in = '0') then
                  fifo_rd_addr <= fifo_rd_addr + 1;
                  data_valid_i <= '1';
                else
                  data_valid_i <= '0';
                end if;
              elsif (fifo_pause_in = '0') then
                if (fifo_dout_i(31 downto 29) = "010") then
                  if (mcc_ee_count_i /= 0) then
                    if (occupancy_count = 1) then
                      data_valid_i <= '1'; -- Sometimes Extra DV on Wrong Header
                    else
                      data_valid_i <= '0'; -- Sometimes NO DV on Trailer
                    end if;
                    fifo_readout <= check_next_word;  --send_mcc_ee;
                  else
                    data_valid_i <= '1';
                    fifo_rd_addr <= fifo_rd_addr + 1;
                  end if;
                else
                  data_valid_i <= '1';
                  fifo_rd_addr <= fifo_rd_addr + 1;
                end if;
              else
                data_valid_i <= '0';
              end if;
--  *************************************************************
--  This section handles all cases when all of the data is in the FIFO
            elsif (rden_in = '1' AND fifo_rd_addr = fifo_wr_addr) then
              if ((num_accepts_in = "0000" OR num_accepts_i = num_accepts_in) AND fifo_dout_i(31 downto 29) = "010") then
                data_valid_i <= '0';
                fifo_eoe_word_out <= '1';
                fifo_readout <= done;
              elsif (fifo_dout_i(31 downto 29) = "010" AND mcc_ee_count_i /= 0) then
                data_valid_i <= '0';  -- Send a trailer
                fifo_readout <= check_next_word;  --send_mcc_ee;
              else
                data_valid_i <= '0';
              end if;
            end if;
--  *************************************************************
          else
            data_valid_i <= '0';
          end if;
 
          if (config_mode_in = '0') then
            if (fifo_dout_i(31 downto 29) = "001" AND data_valid_i = '1') then
              mcc_ee_count_i <= mcc_ee_count_i + fifo_dout_i(15 downto 12);
--              mcc_ee_code_i <= mcc_ee_code_i + fifo_dout_i(15 downto 12);
--              mcc_ee_code_i <= mcc_ee_count_i + fifo_dout_i(15 downto 12);
              if (fifo_dout_i(11 downto 8) < l1id_check_i(3 downto 0) AND fifo_dout_i(15 downto 12) /= X"0") then
                pres_l1id_i <= l1id_check_i(3 downto 0);
              else
                pres_l1id_i <= fifo_dout_i(11 downto 8);
              end if;
              pres_bcid_i <= fifo_dout_i(7 downto 0);
            end if;
          else
            mcc_ee_count_i <= (others => '0');
            mcc_ee_code_i <= (others => '0');
          end if;

          if (fifo_dout_i(31 downto 29) = "010" AND data_valid_i = '1') then
            num_accepts_i <= num_accepts_i + 1 ;
          end if;
          
-- NOT USED YET
          if (mcc_ee_code_i(3 downto 0) = X"F") then
            mcc_ee_overflow <= '1';
          end if;

        when check_next_word =>
          data_valid_i <= '0';
          if (fifo_dout_i(31 downto 29) = "010" AND data_valid_i = '1') then
            num_accepts_i <= num_accepts_i + 1 ;
            fifo_rd_addr <= fifo_rd_addr + 1;
          end if;
          fifo_readout <= send_mcc_ee; 

        when send_mcc_ee => 
          case mcc_empty_event is
            when idle =>
              mcc_empty_event <= check_next_word;
              data_valid_i <= '0';

            when check_next_word =>  -- Check to see if the current header belongs in the next event
              if (to_error_in = '1') then
                mcc_empty_event <= idle;
                fifo_readout <= done;
              elsif (mcc_ee_count_i /= 0) then  -- if there are more empty events, process them
                mcc_empty_event <= send_eeh;
                data_valid_i <= '1';
                if (mcc_ee_count_i > 14) then
                  mcc_ee_code_i <= "01111";
                else
                  mcc_ee_code_i <= mcc_ee_count_i;
                end if;

              elsif (fifo_empty_i = '0') then
                if (fifo_dout_i(31 downto 29) = "001") then
                  if (pres_l1id_i > fifo_dout_i(11 downto 8)) then
                    fifo_readout <= play;  -- Play the next data word from FIFO
                    mcc_empty_event <= idle;
                  elsif (pres_l1id_i = fifo_dout_i(11 downto 8)) then
  -- Check BCID to see if the next event is sequential (protect against 0xFF rollover)
                    if (fifo_dout_i(7 downto 0) > pres_bcid_i + 1) then
                      mcc_empty_event <= send_eeh;  --Send EEH/T
                      data_valid_i <= '1';
                    else
                      fifo_readout <= play;  -- Play the next data word from FIFO
                      mcc_empty_event <= idle;
                    end if;
                  elsif (pres_l1id_i < fifo_dout_i(11 downto 8)) then
                    mcc_empty_event <= send_eeh;  --Send EEH/T
                    data_valid_i <= '1';
                  end if;
                else  -- if not, close out the event
                  mcc_empty_event <= send_eeh;
                  data_valid_i <= '1';
                end if;
              end if;

            when send_eeh =>
              if (fifo_pause_in = '0') then
                data_valid_i <= '1';
                pres_bcid_i <= pres_bcid_i + 1 ;
                if (mcc_ee_count_i > 0) then  -- Guard against 0 to F edge
                  mcc_ee_count_i <= mcc_ee_count_i - '1';
                end if;
               mcc_empty_event <= send_eet;
              else
                data_valid_i <= '0';
              end if;
       
            when send_eet =>
              if (fifo_pause_in = '0') then
                if (num_accepts_in = "0000" OR num_accepts_i = num_accepts_in) then
                  data_valid_i <= '0';
                  mcc_empty_event <= done;
                  fifo_eoe_word_out <= '1';
    -- Added to debug complex cases
    -- If skipped evt = 0 before NA, then one more HT must be added to the evt
                elsif (mcc_ee_count_i = 0) then
                  num_accepts_i <= num_accepts_i + 1 ;
                  data_valid_i <= '0';
                  mcc_empty_event <= check_next_word;
                else
                  mcc_empty_event <= send_eeh;
                  num_accepts_i <= num_accepts_i + 1 ;
                  data_valid_i <= '1';
                end if;
              else
                data_valid_i <= '0';
              end if;

            when others =>
              mcc_empty_event <= idle;
              fifo_readout <= done;
              fifo_eoe_word_out <= '0';
          end case;

        when done =>
          num_accepts_i <= (others => '0');
          mcc_ee_code_i <= (others => '0');
          data_valid_i  <= '0';
          fifo_eoe_word_out <= '0';
          if (rden_in = '0') then
            if (fifo_dout_i(31 downto 29) = "010") then
              fifo_rd_addr <= fifo_rd_addr + 1; -- Flush lingering Trailer
            end if;
            fifo_readout <= idle;
            l1id_check_i <= l1id_check_i + '1';
          end if;

        when others =>
      end case;          
    end if;
  end if;
end process output_counter;

-------------------------------------------------------------------------------
-- fifo word occupancy counter
occupancy_counter : process (
  rst_n_in, 
  clk_in
  )
begin
  if (rst_n_in = '0') then
    occupancy_count <= (others => '0');
  elsif (clk_in'event and clk_in = '1') then
    if (link_enable_in = '1') then
      occupancy_count <= fifo_wr_addr - fifo_rd_addr;
    else
      occupancy_count <= (others => '0');
    end if;
  end if;
end process occupancy_counter;

fifo_empty_flag : process (
  clk_in,
  rst_n_in, 
  occupancy_count,
  fifo_readout,
  mcc_ee_count_i
  )
begin
  if (rst_n_in = '0') then
    fifo_empty_i <= '1';
--  else
  elsif (clk_in'event and clk_in = '1') then
-- FIFO is empty if there is no data and the MCC EE count = 0
-- This will allow the FIFO ROC to send control signals for MCC-EE
    if (occupancy_count = 0) then
      fifo_empty_i <= '1';
    else
      fifo_empty_i <= '0';
    end if;     
  end if;
end process;

fifo_full_flag : process (
  rst_n_in, 
  occupancy_count,
  link_enable_in
  )
begin
  if (rst_n_in = '0') then
    fifo_full_i  <= '0';
  else
    if (link_enable_in = '1' AND occupancy_count = 2046) then
      fifo_full_i <= '1';
    else
      fifo_full_i <= '0';
    end if;     
  end if;
end process;

end rtl;